#!/usr/bin/env python
# -*- coding: ISO-8859-1 -*-
# generated by wxGlade 0.3.5.1 on Thu Apr 21 12:10:56 2005

# Papagayo-NG, a lip-sync tool for use with several different animation suites
# Original Copyright (C) 2005 Mike Clifton
# Contact information at http://www.lostmarble.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

import time

import PySide2.QtGui as QtGui
import PySide2.QtWidgets as QtWidgets
import anytree.util
import numpy as np
from anytree import Node

from LipsyncDoc import *


# from utilities import Worker, WorkerSignals


def normalize(x):
    x = np.asarray(x)
    return ((x - x.min()) / (np.ptp(x))) * 0.8


fill_color = QtGui.QColor(162, 205, 242)
line_color = QtGui.QColor(30, 121, 198)
frame_col = QtGui.QColor(192, 192, 192)
frame_text_col = QtGui.QColor(64, 64, 64)
play_back_col = QtGui.QColor(255, 127, 127)
play_fore_col = QtGui.QColor(209, 102, 121)
play_outline_col = QtGui.QColor(128, 0, 0)
text_col = QtGui.QColor(64, 64, 64)
phrase_fill_col = QtGui.QColor(205, 242, 162)
phrase_outline_col = QtGui.QColor(121, 198, 30)
word_fill_col = QtGui.QColor(242, 205, 162)
word_outline_col = QtGui.QColor(198, 121, 30)
phoneme_fill_col = QtGui.QColor(231, 185, 210)
phoneme_outline_col = QtGui.QColor(173, 114, 146)
font = QtGui.QFont("Swiss", 6)

# default_sample_width = 2
# default_samples_per_frame = 4
default_sample_width = 4
default_samples_per_frame = 2


class SceneWithDrag(QtWidgets.QGraphicsScene):
    def dragEnterEvent(self, e):
        e.acceptProposedAction()

    def dropEvent(self, e):
        # find item at these coordinates
        item = self.itemAt(e.scenePos(), QtGui.QTransform())
        if item:
            if item.setAcceptDrops:
                # pass on event to item at the coordinates
                item.dropEvent(e)
                try:
                    item.dropEvent(e)
                except RuntimeError:
                    pass  # This will suppress a Runtime Error generated when dropping into a widget with no MyProxy

    def dragMoveEvent(self, e):
        e.acceptProposedAction()


class MovableButton(QtWidgets.QPushButton):
    def __init__(self, lipsync_object, wfv_parent, phoneme_offset=None):
        super(MovableButton, self).__init__(lipsync_object.text, None)
        self.title = lipsync_object.text
        self.node = None
        self.phoneme_offset = phoneme_offset
        self.lipsync_object = lipsync_object
        self.style = None
        self.is_resizing = False
        self.is_moving = False
        self.resize_origin = 0  # 0 = left 1 = right
        self.hot_spot = 0
        self.wfv_parent = wfv_parent
        self.setToolTip(lipsync_object.text)
        self.create_and_set_style()
        self.set_tags(self.lipsync_object.tags)
        self.setMinimumWidth(self.convert_to_pixels(1))
        self.fit_text_to_size()

    def text_size(self):
        font_metrics = QtGui.QFontMetrics(self.font())
        return font_metrics.width(self.title)

    def text_fits_in_button(self):
        if not self.is_phoneme():
            return self.text_size() < self.convert_to_pixels(self.get_frame_size()) + self.convert_to_pixels(0.5)
        else:
            return self.text_size() < self.convert_to_pixels(self.get_frame_size()) - self.convert_to_pixels(0.5)

    def fit_text_to_size(self):
        self.title = self.lipsync_object.text
        while not self.text_fits_in_button():
            if len(self.title) > 1:
                self.title = self.title[:-1]
            else:
                break
        self.setText(self.title)

    def create_and_set_style(self):
        if not self.style:
            if self.is_phrase():

                self.style = "QPushButton {{color: #000000; background-color:rgb({0},{1},{2});".format(
                    phrase_fill_col.red(),
                    phrase_fill_col.green(),
                    phrase_fill_col.blue())
                self.style += "background-image: url(:/rsrc/marker.png); "
                self.style += "background-repeat: repeat-y; background-position: right;"
                self.style += "border:1px solid rgb({0},{1},{2});}};".format(phrase_outline_col.red(),
                                                                             phrase_outline_col.green(),
                                                                             phrase_outline_col.blue())
                # self.style = """QPushButton {
                #                 color: #000000;
                #                 border-image: url(./rsrc/testbutton2.png) 2 10 2 2 repeat;
                #                 border-top: 2px transparent;
                #                 border-bottom: 2px transparent;
                #                 border-right: 10px transparent;
                #                 border-left: 2px transparent;
                #                 }"""
            elif self.is_word():
                self.style = "QPushButton {{color: #000000; background-color:rgb({0},{1},{2});".format(
                    word_fill_col.red(),
                    word_fill_col.green(),
                    word_fill_col.blue())
                self.style += "background-image: url(:/rsrc/marker.png); "
                self.style += "background-repeat: repeat-y; background-position: right;"
                self.style += "border:1px solid rgb({0},{1},{2});}};".format(word_outline_col.red(),
                                                                             word_outline_col.green(),
                                                                             word_outline_col.blue())
            elif self.is_phoneme():
                self.style = "QPushButton {{color: #000000; background-color:rgb({0},{1},{2});".format(
                    phoneme_fill_col.red(),
                    phoneme_fill_col.green(),
                    phoneme_fill_col.blue())
                self.style += "border:1px solid rgb({0},{1},{2});}};".format(phoneme_outline_col.red(),
                                                                             phoneme_outline_col.green(),
                                                                             phoneme_outline_col.blue())
            self.setStyleSheet(self.style)

    def is_phoneme(self):
        if self.lipsync_object.is_phoneme:
            return True
        else:
            return False

    def is_word(self):
        try:
            self.lipsync_object.phonemes
        except AttributeError:
            return False
        return True

    def is_phrase(self):
        try:
            self.lipsync_object.words
        except AttributeError:
            return False
        return True

    def object_type(self):
        if self.is_phoneme():
            return "phoneme"
        elif self.is_word():
            return "word"
        elif self.is_phrase():
            return "phrase"
        else:
            return None

    def after_reposition(self):
        if self.is_phoneme():
            self.setGeometry(self.convert_to_pixels(self.lipsync_object.frame), self.y(),
                             self.convert_to_pixels(self.get_frame_size()), self.height())
        else:
            self.setGeometry(self.convert_to_pixels(self.lipsync_object.start_frame), self.y(),
                             self.convert_to_pixels(self.get_frame_size()), self.height())
        self.update()

    def get_min_size(self):
        # An object should be at least be able to contain all it's phonemes since only 1 phoneme per frame is allowed.
        if self.is_phoneme():
            num_of_phonemes = 1
        else:
            num_of_phonemes = 0
            for descendant in self.node.descendants:
                if descendant.name.is_phoneme():
                    num_of_phonemes += 1
        return num_of_phonemes

    def get_frame_size(self):
        if self.is_phoneme():
            return 1
        else:
            return self.lipsync_object.end_frame - self.lipsync_object.start_frame

    def has_shrink_room(self):
        if self.is_phoneme():
            return False
        else:
            if self.get_min_size() >= self.get_frame_size():
                return False
            else:
                return True

    def has_left_sibling(self):
        try:
            left_sibling = bool(self.get_left_sibling())
        except AttributeError:
            left_sibling = False
        return left_sibling

    def has_right_sibling(self):
        try:
            right_sibling = bool(self.get_right_sibling())
        except AttributeError:
            right_sibling = False
        return right_sibling

    def get_left_max(self):
        try:
            temp = self.get_left_sibling().name
            if not temp.lipsync_object.is_phoneme:
                left_most_pos = temp.lipsync_object.end_frame
            else:
                left_most_pos = temp.lipsync_object.frame + 1
        except AttributeError:
            if self.node.depth > 1:
                left_most_pos = self.node.parent.name.lipsync_object.start_frame
            else:
                left_most_pos = 0
        return left_most_pos

    def get_right_max(self):
        try:
            temp = self.get_right_sibling().name
            if not temp.is_phoneme():
                right_most_pos = temp.lipsync_object.start_frame
            else:
                right_most_pos = temp.lipsync_object.frame
        except AttributeError:
            if self.node.depth > 1:
                right_most_pos = self.node.parent.name.lipsync_object.end_frame
            else:
                right_most_pos = self.convert_to_frames(self.wfv_parent.list_of_lines[-1].p2().x())
        return right_most_pos

    def convert_to_pixels(self, frame_pos):
        return frame_pos * self.wfv_parent.frame_width

    def convert_to_frames(self, pixel_pos):
        return pixel_pos / self.wfv_parent.frame_width

    def mouseMoveEvent(self, event):
        if not self.wfv_parent.doc.sound.is_playing():
            if event.buttons() == QtCore.Qt.LeftButton:
                if not self.is_phoneme():
                    if (round(self.convert_to_frames(
                            self.x() + event.x())) + 1 >= self.lipsync_object.end_frame) and not self.is_moving:
                        self.is_resizing = True
                        self.resize_origin = 1
                else:
                    self.is_resizing = False
                    self.is_moving = True
            else:
                self.is_moving = True
            if self.is_resizing and not self.is_moving:
                if self.get_frame_size() < self.get_min_size():
                    self.lipsync_object.end_frame = self.lipsync_object.start_frame + self.get_min_size()
                    self.wfv_parent.doc.dirty = True
                self.after_reposition()
                if self.resize_origin == 1:  # start resize from right side
                    if round(self.convert_to_frames(
                            event.x() + self.x())) + 1 >= self.lipsync_object.start_frame + self.get_min_size():
                        if round(self.convert_to_frames(event.x() + self.x())) + 1 <= self.get_right_max():
                            self.lipsync_object.end_frame = round(self.convert_to_frames(event.x() + self.x())) + 1
                            self.wfv_parent.doc.dirty = True
                            self.resize(self.convert_to_pixels(self.lipsync_object.end_frame) -
                                        self.convert_to_pixels(self.lipsync_object.start_frame), self.height())
                # elif self.resize_origin == 0:  # start resize from left side
                #     if round(self.convert_to_frames(event.x() + self.x())) < self.lipsync_object.end_frame:
                #         if round(self.convert_to_frames(event.x() + self.x())) >= self.get_left_max():
                #             self.lipsync_object.start_frame = round(self.convert_to_frames(event.x() + self.x()))
                #             new_length = self.convert_to_pixels(self.lipsync_object.end_frame) - self.convert_to_pixels(self.lipsync_object.start_frame)
                #             self.resize(new_length, self.height())
                #             self.move(self.convert_to_pixels(self.lipsync_object.start_frame), self.y())
            else:
                self.is_moving = True
                mime_data = QtCore.QMimeData()
                drag = QtGui.QDrag(self)
                drag.setMimeData(mime_data)
                drag.setHotSpot(event.pos() - self.rect().topLeft())
                self.hot_spot = drag.hotSpot().x()
                # PyQt5 and PySide use different function names here, likely a Qt4 vs Qt5 problem.
                try:
                    exec("dropAction = drag.exec(QtCore.Qt.MoveAction)")
                except (SyntaxError, AttributeError):
                    dropAction = drag.start(QtCore.Qt.MoveAction)

    def mousePressEvent(self, event):
        if not self.wfv_parent.doc.sound.is_playing():
            if event.button() == QtCore.Qt.RightButton and self.is_word():
                # manually enter the pronunciation for this word
                dlg = PronunciationDialog(self, self.wfv_parent.doc.parent.phonemeset.set)
                dlg.word_label.setText(dlg.word_label.text() + ' ' + self.text())
                dlg.setWindowTitle(self.title)
                prev_phoneme_list = ""
                for p in self.node.children:
                    prev_phoneme_list += " " + p.name.lipsync_object.text
                dlg.phoneme_ctrl.setText(prev_phoneme_list)
                if dlg.exec_():
                    list_of_new_phonemes = dlg.phoneme_ctrl.text().split()
                    if list_of_new_phonemes:
                        if list_of_new_phonemes != prev_phoneme_list.split():
                            old_childnodes = self.node.children
                            for old_node in old_childnodes:
                                for proxy in self.wfv_parent.items():
                                    try:
                                        if proxy.widget() == old_node.name:
                                            self.wfv_parent.scene().removeItem(proxy)
                                    except AttributeError:
                                        pass
                            old_childnodes = []
                            self.node.children = []
                            self.lipsync_object.phonemes = []
                            font_metrics = QtGui.QFontMetrics(font)
                            text_width, text_height = font_metrics.width("Ojyg"), font_metrics.height() + 6
                            for phoneme_count, p in enumerate(list_of_new_phonemes):
                                phoneme = LipsyncPhoneme()
                                phoneme.text = p
                                phoneme.frame = self.lipsync_object.start_frame + phoneme_count
                                self.lipsync_object.phonemes.append(phoneme)
                                temp_button = MovableButton(phoneme, self.wfv_parent, phoneme_count % 2)
                                temp_button.node = Node(temp_button, parent=self.node)
                                temp_scene_widget = self.wfv_parent.scene().addWidget(temp_button)
                                # temp_scene_widget.setParent(self.wfv_parent)
                                temp_rect = QtCore.QRect(phoneme.frame * self.wfv_parent.frame_width,
                                                         self.wfv_parent.height() -
                                                         (self.wfv_parent.horizontalScrollBar().height() * 1.5) -
                                                         (text_height + (text_height * (phoneme_count % 2))),
                                                         self.wfv_parent.frame_width, text_height)
                                temp_scene_widget.setGeometry(temp_rect)
                                temp_scene_widget.setZValue(99)
                            self.wfv_parent.doc.dirty = True

    def mouseDoubleClickEvent(self, event):
        if not self.wfv_parent.doc.sound.is_playing() and not self.is_phoneme():
            start = self.lipsync_object.start_frame / self.wfv_parent.doc.fps
            length = (self.lipsync_object.end_frame - self.lipsync_object.start_frame) / self.wfv_parent.doc.fps
            self.wfv_parent.doc.sound.play_segment(start, length)
            old_cur_frame = 0
            start_time = 0
            self.wfv_parent.temp_play_marker.setVisible(True)
            main_window = self.wfv_parent.parentWidget().parentWidget().parentWidget()  # lol
            main_window.action_stop.setEnabled(True)
            main_window.action_play.setEnabled(False)
            while self.wfv_parent.doc.sound.is_playing():
                QtCore.QCoreApplication.processEvents()
                cur_frame = int(self.wfv_parent.doc.sound.current_time() * self.wfv_parent.doc.fps)
                if old_cur_frame != cur_frame:
                    old_cur_frame = cur_frame

                    main_window.mouth_view.set_frame(old_cur_frame)
                    self.wfv_parent.set_frame(old_cur_frame)
                    try:
                        fps = 1.0 / (time.time() - start_time)
                    except ZeroDivisionError:
                        fps = 60
                    main_window.statusbar.showMessage("Frame: {:d} FPS: {:d}".format((cur_frame + 1), int(fps)))
                    self.wfv_parent.scroll_position = self.wfv_parent.horizontalScrollBar().value()
                    start_time = time.time()
                self.wfv_parent.update()
            self.wfv_parent.temp_play_marker.setVisible(False)
            main_window.action_stop.setEnabled(False)
            main_window.action_play.setEnabled(True)
            main_window.statusbar.showMessage("Stopped")
            main_window.waveform_view.horizontalScrollBar().setValue(main_window.waveform_view.scroll_position)
            main_window.waveform_view.update()

    def mouseReleaseEvent(self, event):
        if self.is_moving:
            self.is_moving = False
        if self.is_resizing:
            self.reposition_descendants2(True)
            self.is_resizing = False

    def set_tags(self, new_taglist):
        self.lipsync_object.tags = new_taglist
        self.setToolTip("".join("{}\n".format(entry) for entry in self.lipsync_object.tags)[:-1])
        # Change the border-style or something like that depending on whether there are tags or not
        if len(self.lipsync_object.tags) > 0:
            if "solid" in self.styleSheet():
                self.setStyleSheet(self.styleSheet().replace("solid", "dashed "))
        else:
            if "dashed " in self.styleSheet():
                self.setStyleSheet(self.styleSheet().replace("dashed ", "solid"))

    def reposition_descendants(self, did_resize=False, x_diff=0):
        if did_resize:
            for child in self.node.children:
                child.name.reposition_to_left()
        else:
            for child in self.node.descendants:
                if child.name.is_phoneme():
                    child.name.lipsync_object.frame += x_diff
                else:
                    child.name.lipsync_object.start_frame += x_diff
                    child.name.lipsync_object.end_frame += x_diff
                child.name.after_reposition()
            self.wfv_parent.doc.dirty = True

    def reposition_descendants2(self, did_resize=False, x_diff=0):
        if did_resize:
            if self.is_word():
                for position, child in enumerate(self.node.children):
                    child.name.lipsync_object.frame = round(self.lipsync_object.start_frame +
                                                            ((self.get_frame_size() / self.get_min_size()) * position))
                    child.name.after_reposition()
                self.wfv_parent.doc.dirty = True
            elif self.is_phrase():
                extra_space = self.get_frame_size() - self.get_min_size()
                for child in self.node.children:
                    if child.name.has_left_sibling():
                        child.name.lipsync_object.start_frame = child.name.get_left_sibling().name.lipsync_object.end_frame
                        child.name.lipsync_object.end_frame = child.name.lipsync_object.start_frame + child.name.get_min_size()
                    else:
                        child.name.lipsync_object.start_frame = child.name.node.parent.name.lipsync_object.start_frame
                        child.name.lipsync_object.end_frame = child.name.lipsync_object.start_frame + child.name.get_min_size()
                last_position = -1
                moved_child = False
                while extra_space > 0:
                    if last_position == len(self.node.children) - 1:
                        last_position = -1
                    if not moved_child:
                        last_position = -1
                    moved_child = False
                    for position, child in enumerate(self.node.children):
                        if child.name.has_left_sibling():
                            if child.name.lipsync_object.start_frame < child.name.get_left_sibling().name.lipsync_object.end_frame:
                                child.name.lipsync_object.start_frame += 1
                                child.name.lipsync_object.end_frame += 1
                            else:
                                if extra_space and not moved_child and (position > last_position):
                                    child.name.lipsync_object.end_frame += 1
                                    extra_space -= 1
                                    moved_child = True
                                    last_position = position
                        else:
                            if extra_space and not moved_child and (position > last_position):
                                child.name.lipsync_object.end_frame += 1
                                extra_space -= 1
                                moved_child = True
                                last_position = position
                    if not moved_child and extra_space == 0:
                        break
                for child in self.node.children:
                    child.name.after_reposition()
                    child.name.reposition_descendants2(True, 0)
                self.wfv_parent.doc.dirty = True
        else:
            for child in self.node.descendants:
                if child.name.is_phoneme():
                    child.name.lipsync_object.frame += x_diff
                else:
                    child.name.lipsync_object.start_frame += x_diff
                    child.name.lipsync_object.end_frame += x_diff
                child.name.after_reposition()
                self.wfv_parent.doc.dirty = True

    def reposition_to_left(self):
        if self.has_left_sibling():
            if self.is_phoneme():
                self.lipsync_object.frame = self.get_left_sibling().name.lipsync_object.frame + 1
            else:
                self.lipsync_object.start_frame = self.get_left_sibling().name.lipsync_object.end_frame
                self.lipsync_object.end_frame = self.lipsync_object.start_frame + self.get_min_size()
                for child in self.node.children:
                    child.name.reposition_to_left()
        else:
            if self.is_phoneme():
                self.lipsync_object.frame = self.node.parent.name.lipsync_object.start_frame
            else:
                self.lipsync_object.start_frame = self.node.parent.name.lipsync_object.start_frame
                self.lipsync_object.end_frame = self.lipsync_object.start_frame + self.get_min_size()
                for child in self.node.children:
                    child.name.reposition_to_left()
        self.after_reposition()
        self.wfv_parent.doc.dirty = True

    def get_left_sibling(self):
        return anytree.util.leftsibling(self.node)

    def get_right_sibling(self):
        return anytree.util.rightsibling(self.node)

    def get_parent(self):
        if self.object_type() != "phrase":
            return self.node.parent
        else:
            return None

    def __del__(self):
        try:
            self.deleteLater()
        except RuntimeError:
            pass


class WaveformView(QtWidgets.QGraphicsView):
    def __init__(self, parent=None):
        super(WaveformView, self).__init__(parent)
        self.setScene(SceneWithDrag(self))
        self.setVerticalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)
        self.setHorizontalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOn)
        self.setViewportUpdateMode(QtWidgets.QGraphicsView.NoViewportUpdate)
        self.setAcceptDrops(True)
        self.setMouseTracking(True)
        # Other initialization
        self.main_window = self.parentWidget().parentWidget().parentWidget()  # lol
        self.doc = None
        self.currently_selected_object = None
        self.is_scrubbing = False
        self.cur_frame = 0
        self.old_frame = 0
        self.default_sample_width = default_sample_width
        self.default_samples_per_frame = default_samples_per_frame
        self.sample_width = self.default_sample_width
        self.samples_per_frame = self.default_samples_per_frame
        self.samples_per_sec = 24 * self.samples_per_frame
        self.frame_width = self.sample_width * self.samples_per_frame
        self.phrase_bottom = 16
        self.word_bottom = 32
        self.phoneme_top = 128
        self.waveform_polygon = None
        self.wv_height = 1
        self.temp_phrase = None
        self.temp_word = None
        self.temp_phoneme = None
        self.temp_button = None
        self.draw_play_marker = False
        self.num_samples = 0
        self.list_of_lines = []
        self.amp = []
        self.temp_play_marker = None
        self.scroll_position = 0
        self.first_update = True
        self.node = None
        self.did_resize = None
        self.main_node = None
        self.threadpool = QtCore.QThreadPool.globalInstance()
        self.scene().setSceneRect(0, 0, self.width(), self.height())
        self.resize_timer = QtCore.QTimer(self)
        self.resize_timer.setSingleShot(True)
        self.connect(self.resize_timer, QtCore.SIGNAL("timeout()"), self.resize_finished)

        print("LoadedWaveFormView")

    def dropEvent(self, event):
        print("DragLeave")  # Strangely no dragLeaveEvent fires but a dropEvent instead...
        if event.mimeData().hasUrls():
            # event.accept()
            for url in event.mimeData().urls():
                if sys.platform == "darwin":
                    from Foundation import NSURL
                    fname = str(NSURL.URLWithString_(str(url.toString())).filePathURL().path())
                    self.topLevelWidget().lip_sync_frame.open(fname)
                else:
                    fname = str(url.toLocalFile())
                    self.topLevelWidget().lip_sync_frame.open(fname)
            return True
        else:
            if event.source():
                event.source().is_moving = False
            event.accept()

    def dragEnterEvent(self, e):
        print("DragEnter!")
        e.accept()

    def mousePressEvent(self, event):
        if event.button() == QtCore.Qt.LeftButton:
            possible_item = self.itemAt(event.pos())
            if type(possible_item) == QtWidgets.QGraphicsPolygonItem:
                possible_item = None
            if not possible_item:
                if self.currently_selected_object:
                    try:
                        new_style = self.currently_selected_object.styleSheet()
                        if "2px" in new_style:
                            new_style = new_style.replace("2px", "1px")
                        else:
                            pass
                        self.currently_selected_object.setStyleSheet(new_style)
                    except RuntimeError:
                        pass  # The real object was deleted, instead of carefully tracking we simply do this
                self.currently_selected_object = None
                self.main_window.list_of_tags.clear()
                self.main_window.tag_list_group.setEnabled(False)
                self.main_window.tag_list_group.setTitle("Selected Object Tags")
                self.main_window.parent_tags.clear()
                self.main_window.parent_tags.setEnabled(False)
                self.is_scrubbing = True
            else:
                self.main_window.tag_list_group.setEnabled(True)
                if self.currently_selected_object:
                    try:
                        new_style = self.currently_selected_object.styleSheet()
                        if "2px" in new_style:
                            new_style = new_style.replace("2px", "1px")
                        else:
                            pass
                        self.currently_selected_object.setStyleSheet(new_style)
                    except RuntimeError:
                        pass  # The real object was deleted, instead of carefully tracking we simply do this
                self.currently_selected_object = possible_item.widget()
                new_style = self.currently_selected_object.styleSheet()
                if "1px" in new_style:
                    new_style = new_style.replace("1px", "2px")
                else:
                    pass
                self.currently_selected_object.setStyleSheet(new_style)
                self.main_window.list_of_tags.clear()
                self.main_window.list_of_tags.addItems(self.currently_selected_object.lipsync_object.tags)
                title_part_two = self.currently_selected_object.title
                if len(self.currently_selected_object.title) > 40:
                    title_part_two = self.currently_selected_object.title[0:40] + "..."
                new_title = self.currently_selected_object.object_type().title() + ": " + title_part_two
                self.main_window.tag_list_group.setTitle(new_title)
                self.main_window.parent_tags.clear()
                self.main_window.parent_tags.setEnabled(False)
                if self.currently_selected_object.object_type() == "phoneme":
                    parent_word = self.currently_selected_object.get_parent().name
                    parent_phrase = parent_word.get_parent().name
                    word_tags = parent_word.lipsync_object.tags
                    phrase_tags = parent_phrase.lipsync_object.tags
                    if word_tags or phrase_tags:
                        self.main_window.parent_tags.setEnabled(True)
                    if phrase_tags:
                        list_of_phrase_tags = []
                        for tag in phrase_tags:
                            new_tag = QtWidgets.QTreeWidgetItem([tag])
                            list_of_phrase_tags.append(new_tag)
                        phrase_tree = QtWidgets.QTreeWidgetItem(["Phrase: " + parent_phrase.title])
                        phrase_tree.addChildren(list_of_phrase_tags)
                        self.main_window.parent_tags.addTopLevelItem(phrase_tree)
                        phrase_tree.setExpanded(True)
                    if word_tags:
                        list_of_word_tags = []
                        for tag in word_tags:
                            new_tag = QtWidgets.QTreeWidgetItem([tag])
                            list_of_word_tags.append(new_tag)
                        word_tree = QtWidgets.QTreeWidgetItem(["Word: " + parent_word.title])
                        word_tree.addChildren(list_of_word_tags)
                        self.main_window.parent_tags.addTopLevelItem(word_tree)
                        word_tree.setExpanded(True)
                elif self.currently_selected_object.object_type() == "word":
                    parent_phrase = self.currently_selected_object.get_parent().name
                    parent_tags = parent_phrase.lipsync_object.tags
                    list_of_tags = []
                    if parent_tags:
                        self.main_window.parent_tags.setEnabled(True)
                        for tag in parent_tags:
                            new_tag = QtWidgets.QTreeWidgetItem([tag])
                            list_of_tags.append(new_tag)
                        phrase_tree = QtWidgets.QTreeWidgetItem(["Phrase: " + parent_phrase.title])
                        phrase_tree.addChildren(list_of_tags)
                        self.main_window.parent_tags.addTopLevelItem(phrase_tree)
                        phrase_tree.setExpanded(True)
                else:
                    self.main_window.parent_tags.setEnabled(False)
        event.accept()
        super(WaveformView, self).mousePressEvent(event)

    def mouseReleaseEvent(self, event):
        if self.is_scrubbing:
            self.is_scrubbing = False
            self.doc.sound.stop()
            self.temp_play_marker.setVisible(False)
            self.main_window.mouth_view.set_frame(0)
        super(WaveformView, self).mouseReleaseEvent(event)

    def mouseMoveEvent(self, event):
        if self.is_scrubbing:
            mouse_scene_pos = self.mapToScene(event.pos()).x()
            if not self.doc.sound.is_playing():
                start = round(mouse_scene_pos / self.frame_width) / self.doc.fps
                length = self.frame_width / self.doc.fps
                self.doc.sound.play_segment(start, length)
            self.draw_play_marker = True
            self.temp_play_marker.setVisible(True)
            self.temp_play_marker.setPos(round(mouse_scene_pos / self.frame_width) * self.frame_width, 0)
            self.main_window.mouth_view.set_frame(round(mouse_scene_pos / self.frame_width))
        else:
            super(WaveformView, self).mouseMoveEvent(event)

    def dragMoveEvent(self, e):
        if not self.doc.sound.is_playing():
            if e.source():
                position = e.pos()
                if self.width() > self.sceneRect().width():
                    new_x = e.pos().x() + self.horizontalScrollBar().value() - \
                            ((self.width() - self.sceneRect().width()) / 2) - e.source().hot_spot
                else:
                    new_x = e.pos().x() + self.horizontalScrollBar().value() - e.source().hot_spot
                dropped_widget = e.source()
                if new_x >= dropped_widget.get_left_max() * self.frame_width:
                    if new_x + dropped_widget.width() <= dropped_widget.get_right_max() * self.frame_width:
                        x_diff = 0
                        dropped_widget.move(new_x, dropped_widget.y())
                        # after moving save the position and align to the grid based on that. Hacky but works!
                        if dropped_widget.lipsync_object.is_phoneme:
                            x_diff = round(dropped_widget.x() / self.frame_width) - dropped_widget.lipsync_object.frame
                            dropped_widget.lipsync_object.frame = round(new_x / self.frame_width)
                            dropped_widget.move(dropped_widget.lipsync_object.frame * self.frame_width,
                                                dropped_widget.y())
                        else:
                            x_diff = round(
                                dropped_widget.x() / self.frame_width) - dropped_widget.lipsync_object.start_frame
                            dropped_widget.lipsync_object.start_frame = round(dropped_widget.x() / self.frame_width)
                            dropped_widget.lipsync_object.end_frame = round(
                                (dropped_widget.x() + dropped_widget.width()) / self.frame_width)
                            dropped_widget.move(dropped_widget.lipsync_object.start_frame * self.frame_width,
                                                dropped_widget.y())
                            # Move the children!
                            dropped_widget.reposition_descendants(False, x_diff)
                        self.doc.dirty = True
        e.accept()

    def set_frame(self, frame):
        if self.temp_play_marker not in self.scene().items():
            self.temp_play_marker = self.scene().addRect(0, 1, self.frame_width + 1, self.height(),
                                                         QtGui.QPen(play_outline_col),
                                                         QtGui.QBrush(play_fore_col, QtCore.Qt.SolidPattern))
            self.temp_play_marker.setZValue(1000)
            self.temp_play_marker.setOpacity(0.5)
            self.temp_play_marker.setVisible(True)
        self.centerOn(self.temp_play_marker)
        self.temp_play_marker.setPos(frame * self.frame_width, 0)
        self.update()
        self.scene().update()

    def drawBackground(self, painter, rect):
        background_brush = QtGui.QBrush(QtGui.QColor(255, 255, 255), QtCore.Qt.SolidPattern)
        painter.fillRect(rect, background_brush)
        if self.doc is not None:
            pen = QtGui.QPen(frame_col)
            # pen.setWidth(5)
            painter.setPen(pen)
            painter.setFont(font)

            first_sample = 0
            last_sample = len(self.amp)
            bg_height = self.height() + self.horizontalScrollBar().height()
            half_client_height = bg_height / 2
            font_metrics = QtGui.QFontMetrics(font)
            text_width, top_border = font_metrics.width("Ojyg"), font_metrics.height() * 2
            x = first_sample * self.sample_width
            frame = first_sample / self.samples_per_frame
            fps = int(round(self.doc.fps))
            sample = first_sample
            self.list_of_lines = []
            list_of_textmarkers = []
            for i in range(int(first_sample), int(last_sample)):
                if (i + 1) % self.samples_per_frame == 0:
                    frame_x = (frame + 1) * self.frame_width
                    if (self.frame_width > 2) or ((frame + 1) % fps == 0):
                        self.list_of_lines.append(QtCore.QLineF(frame_x, top_border, frame_x, bg_height))
                    # draw frame label
                    if (self.frame_width > 30) or ((int(frame) + 1) % 5 == 0):
                        self.list_of_lines.append(QtCore.QLineF(frame_x, 0, frame_x, top_border))
                        self.list_of_lines.append(QtCore.QLineF(frame_x + 1, 0, frame_x + 1, bg_height))
                        temp_rect = QtCore.QRectF(int(frame_x + 4), font_metrics.height() - 2, text_width, top_border)
                        # Positioning is a bit different in QT here
                        list_of_textmarkers.append((temp_rect, str(int(frame + 1))))
                x += self.sample_width
                sample += 1
                if sample % self.samples_per_frame == 0:
                    frame += 1
            painter.drawLines(self.list_of_lines)
            for text_marker in list_of_textmarkers:
                painter.drawText(text_marker[0], QtCore.Qt.AlignLeft, text_marker[1])

    def start_create_waveform(self):
        worker = Worker(self.create_waveform)
        worker.signals.finished.connect(self.waveform_finished)
        worker.signals.progress.connect(self.topLevelWidget().lip_sync_frame.status_bar_progress)

        self.topLevelWidget().lip_sync_frame.status_progress.show()
        available_height = self.height() / 2
        fitted_samples = self.amp * available_height
        self.topLevelWidget().lip_sync_frame.status_progress.setMaximum(len(fitted_samples))
        self.threadpool.start(worker)
        self.threadpool.waitForDone()

    def waveform_finished(self):
        self.topLevelWidget().lip_sync_frame.status_progress.hide()
        update_rect = self.scene().sceneRect()
        width_factor = 1  # Only the height needs to change.
        height_factor = 1
        update_rect.setHeight(self.size().height() - 1)
        if self.doc:
            update_rect.setWidth(self.waveform_polygon.polygon().boundingRect().width())
            self.setSceneRect(update_rect)
            self.scene().setSceneRect(update_rect)
            origin_x, origin_y = 0, 0
            height_factor = height_factor * self.waveform_polygon.transform().m22()  # We need to add the factors
            self.waveform_polygon.setTransform(QtGui.QTransform().translate(
                origin_x, origin_y).scale(width_factor, height_factor).translate(-origin_x, -origin_y))
            # We need to at least update the Y Position of the Phonemes
            font_metrics = QtGui.QFontMetrics(font)
            text_width, top_border = font_metrics.width("Ojyg"), font_metrics.height() * 2
            text_width, text_height = font_metrics.width("Ojyg"), font_metrics.height() + 6
            top_border += 4
            if self.main_node:
                for phoneme_node in self.main_node.leaves:  # this should be all phonemes
                    widget = phoneme_node.name
                    if widget:
                        if widget.lipsync_object.is_phoneme:  # shouldn't be needed, just to be sure
                            widget.setGeometry(widget.x(), self.height() - (self.horizontalScrollBar().height() * 1.5) -
                                               (text_height + (text_height * widget.phoneme_offset)),
                                               self.frame_width + 5,
                                               text_height)
        self.horizontalScrollBar().setValue(self.scroll_position)
        try:
            if self.temp_play_marker:
                self.temp_play_marker.setRect(self.temp_play_marker.rect().x(), 1, self.frame_width + 1, self.height())
        except RuntimeError:
            pass  # When changing a file we get a RuntimeError from QT because it deletes the temp_play_marker
        self.scene().update()

    def create_waveform(self, progress_callback):
        if self.waveform_polygon in self.scene().items():
            self.scene().removeItem(self.waveform_polygon)
        available_height = self.height() / 2
        fitted_samples = self.amp * available_height
        offset = 0  # available_height / 2
        temp_polygon = QtGui.QPolygonF()
        main_window = self.topLevelWidget()
        for x, y in enumerate(fitted_samples):
            progress_callback.emit((x / 2))
            main_window.statusbar.showMessage(
                "Preparing Waveform: {0}%".format(str(int(((x / 2) / len(fitted_samples)) * 100))))
            temp_polygon.append(QtCore.QPointF(x * self.sample_width, available_height - y + offset))
            if x < len(fitted_samples):
                temp_polygon.append(QtCore.QPointF((x + 1) * self.sample_width, available_height - y + offset))
        for x, y in enumerate(fitted_samples[::-1]):
            progress_callback.emit((len(fitted_samples) / 2) + (x / 2))
            main_window.statusbar.showMessage(
                "Preparing Waveform: {0}%".format(str(int(((x / 2) / len(fitted_samples)) * 100) + 50)))
            temp_polygon.append(QtCore.QPointF((len(fitted_samples) - x) * self.sample_width,
                                               available_height + y + offset))
            if x > 0:
                temp_polygon.append(QtCore.QPointF((len(fitted_samples) - x - 1) * self.sample_width,
                                                   available_height + y + offset))
        self.waveform_polygon = self.scene().addPolygon(temp_polygon, line_color, fill_color)
        self.waveform_polygon.setZValue(1)
        main_window.statusbar.showMessage("Papagayo-NG")

    def start_create_movbuttons(self):
        if self.doc is not None:
            worker = Worker(self.create_movbuttons)
            worker.signals.finished.connect(self.movbuttons_finished)
            worker.signals.progress.connect(self.topLevelWidget().lip_sync_frame.status_bar_progress)

            self.topLevelWidget().lip_sync_frame.status_progress.show()
            self.topLevelWidget().lip_sync_frame.status_progress.setMaximum(self.doc.current_voice.num_children)
            self.threadpool.start(worker)
            self.threadpool.waitForDone()

    def movbuttons_finished(self):
        self.topLevelWidget().lip_sync_frame.status_progress.hide()
        self.start_recalc()

    def create_movbuttons(self, progress_callback):
        if self.doc is not None:
            self.setUpdatesEnabled(False)
            font_metrics = QtGui.QFontMetrics(font)
            text_width, top_border = font_metrics.width("Ojyg"), font_metrics.height() * 2
            text_width, text_height = font_metrics.width("Ojyg"), font_metrics.height() + 6
            top_border += 4
            main_window = self.parentWidget().parentWidget().parentWidget()
            current_num = 0
            self.main_node = Node(
                self.doc.current_voice.text)  # Not actually needed, but should make everything a bit easier
            for phrase in self.doc.current_voice.phrases:
                self.temp_button = MovableButton(phrase, self)
                self.temp_button.node = Node(self.temp_button, parent=self.main_node)
                temp_scene_widget = self.scene().addWidget(self.temp_button)
                temp_rect = QtCore.QRect(phrase.start_frame * self.frame_width, top_border,
                                         (phrase.end_frame - phrase.start_frame) * self.frame_width + 1, text_height)
                temp_scene_widget.setGeometry(temp_rect)
                temp_scene_widget.setZValue(99)
                self.temp_phrase = self.temp_button
                word_count = 0
                current_num += 1
                progress_callback(current_num)
                if self.doc.current_voice.num_children:
                    main_window.statusbar.showMessage("Preparing Buttons: {0}%".format(
                        str(int((current_num / self.doc.current_voice.num_children) * 100))))
                for word in phrase.words:
                    self.temp_button = MovableButton(word, self)
                    self.temp_button.node = Node(self.temp_button, parent=self.temp_phrase.node)
                    temp_scene_widget = self.scene().addWidget(self.temp_button)
                    temp_rect = QtCore.QRect(word.start_frame * self.frame_width, top_border + 4 + text_height +
                                             (text_height * (word_count % 2)), (word.end_frame - word.start_frame) *
                                             self.frame_width + 1, text_height)
                    temp_scene_widget.setGeometry(temp_rect)
                    temp_scene_widget.setZValue(99)
                    self.temp_word = self.temp_button
                    word_count += 1
                    phoneme_count = 0
                    current_num += 1
                    progress_callback(current_num)
                    if self.doc.current_voice.num_children:
                        main_window.statusbar.showMessage("Preparing Buttons: {0}%".format(
                            str(int((current_num / self.doc.current_voice.num_children) * 100))))
                    for phoneme in word.phonemes:
                        self.temp_button = MovableButton(phoneme, self, phoneme_count % 2)
                        self.temp_button.node = Node(self.temp_button, parent=self.temp_word.node)
                        temp_scene_widget = self.scene().addWidget(self.temp_button)
                        temp_rect = QtCore.QRect(phoneme.frame * self.frame_width, self.height() -
                                                 (self.horizontalScrollBar().height() * 1.5) -
                                                 (text_height + (text_height * (phoneme_count % 2))),
                                                 self.frame_width, text_height)
                        temp_scene_widget.setGeometry(temp_rect)
                        temp_scene_widget.setZValue(99)
                        self.temp_phoneme = self.temp_button
                        phoneme_count += 1
                        current_num += 1
                        progress_callback(current_num)
                        if self.doc.current_voice.num_children:
                            main_window.statusbar.showMessage(
                                "Preparing Buttons: {0}%".format(
                                    str(int((current_num / self.doc.current_voice.num_children) * 100))))
            main_window.statusbar.showMessage("Papagayo-NG")
            self.setUpdatesEnabled(True)

    def start_recalc(self, wait_for_done=True):
        worker = Worker(self.recalc_waveform)
        worker.signals.finished.connect(self.recalc_finished)
        worker.signals.progress.connect(self.topLevelWidget().lip_sync_frame.status_bar_progress)

        self.topLevelWidget().lip_sync_frame.status_progress.show()
        self.topLevelWidget().lip_sync_frame.status_progress.setMaximum(self.doc.sound.Duration())
        self.threadpool.start(worker)
        if wait_for_done:
            self.threadpool.waitForDone()

    def recalc_finished(self):
        self.topLevelWidget().lip_sync_frame.status_progress.hide()
        self.start_create_waveform()

    def recalc_waveform(self, progress_callback):
        duration = self.doc.sound.Duration()
        time_pos = 0.0
        sample_dur = 1.0 / self.samples_per_sec
        max_amp = 0.0
        self.amp = []
        while time_pos < duration:
            progress_callback.emit(time_pos)
            self.num_samples += 1
            amp = self.doc.sound.GetRMSAmplitude(time_pos, sample_dur)
            self.amp.append(amp)
            max_amp = max(max_amp, amp)
            time_pos += sample_dur
        self.amp = normalize(self.amp)

    def set_document(self, document, force=False):
        if document != self.doc or force:
            self.doc = document
            if (self.doc is not None) and (self.doc.sound is not None):
                self.scene().clear()
                self.scene().update()
                self.create_movbuttons(self.topLevelWidget().lip_sync_frame.status_bar_progress)
                self.start_recalc()
                if self.temp_play_marker not in self.scene().items():
                    self.temp_play_marker = self.scene().addRect(0, 1, self.frame_width + 1, self.height(),
                                                                 QtGui.QPen(play_outline_col),
                                                                 QtGui.QBrush(play_fore_col, QtCore.Qt.SolidPattern))
                    self.temp_play_marker.setZValue(1000)
                    self.temp_play_marker.setOpacity(0.5)
                    self.temp_play_marker.setVisible(False)
                self.setViewportUpdateMode(QtWidgets.QGraphicsView.FullViewportUpdate)

    def on_slider_change(self, value):
        self.scroll_position = value

    def wheelEvent(self, event):
        self.scroll_position = self.horizontalScrollBar().value() + (event.delta() / 1.2)
        self.horizontalScrollBar().setValue(self.scroll_position)

    def resize_finished(self):
        self.start_create_waveform()

    def resizeEvent(self, event):
        update_rect = self.scene().sceneRect()
        width_factor = 1  # Only the height needs to change.
        try:
            height_factor = event.size().height() / event.oldSize().height()
        except ZeroDivisionError:
            height_factor = 1
        update_rect.setHeight(event.size().height())
        if self.doc:
            update_rect.setWidth(self.waveform_polygon.polygon().boundingRect().width())
            self.setSceneRect(update_rect)
            self.scene().setSceneRect(update_rect)
            origin_x, origin_y = 0, 0
            height_factor = height_factor * self.waveform_polygon.transform().m22()  # We need to add the factors
            self.waveform_polygon.setTransform(QtGui.QTransform().translate(
                origin_x, origin_y).scale(width_factor, height_factor).translate(-origin_x, -origin_y))
            # We need to at least update the Y Position of the Phonemes
            font_metrics = QtGui.QFontMetrics(font)
            text_width, top_border = font_metrics.width("Ojyg"), font_metrics.height() * 2
            text_width, text_height = font_metrics.width("Ojyg"), font_metrics.height() + 6
            top_border += 4
            for phoneme_node in self.main_node.leaves:  # this should be all phonemes
                widget = phoneme_node.name
                if widget:
                    if widget.lipsync_object.is_phoneme:  # shouldn't be needed, just to be sure
                        widget.setGeometry(widget.x(), self.height() - (self.horizontalScrollBar().height() * 1.5) -
                                           (text_height + (text_height * widget.phoneme_offset)), self.frame_width + 5,
                                           text_height)
            self.resize_timer.start(150)
        self.horizontalScrollBar().setValue(self.scroll_position)
        if self.temp_play_marker:
            self.temp_play_marker.setRect(self.temp_play_marker.rect().x(), 1, self.frame_width + 1, self.height())

    def on_zoom_in(self, event=None):
        if (self.doc is not None) and (self.samples_per_frame < 16):
            self.samples_per_frame *= 2
            self.samples_per_sec = self.doc.fps * self.samples_per_frame
            self.frame_width = self.sample_width * self.samples_per_frame
            for node in self.main_node.descendants:
                node.name.after_reposition()
                node.name.fit_text_to_size()
            self.start_recalc()
            if self.temp_play_marker:
                self.temp_play_marker.setRect(self.temp_play_marker.rect().x(), 1, self.frame_width + 1, self.height())
            self.scene().setSceneRect(self.scene().sceneRect().x(), self.scene().sceneRect().y(),
                                      self.sceneRect().width() * 2, self.scene().sceneRect().height())
            self.setSceneRect(self.scene().sceneRect())
            self.scroll_position *= 2
            self.horizontalScrollBar().setValue(self.scroll_position)

    def on_zoom_out(self, event=None):
        if (self.doc is not None) and (self.samples_per_frame > 1):
            self.samples_per_frame /= 2
            self.samples_per_sec = self.doc.fps * self.samples_per_frame
            self.frame_width = self.sample_width * self.samples_per_frame
            for node in self.main_node.descendants:
                node.name.after_reposition()
                node.name.fit_text_to_size()
            self.start_recalc()
            if self.temp_play_marker:
                self.temp_play_marker.setRect(self.temp_play_marker.rect().x(), 1, self.frame_width + 1, self.height())
            self.scene().setSceneRect(self.scene().sceneRect().x(), self.scene().sceneRect().y(),
                                      self.scene().sceneRect().width() / 2, self.scene().sceneRect().height())
            self.setSceneRect(self.scene().sceneRect())
            self.scroll_position /= 2
            self.horizontalScrollBar().setValue(self.scroll_position)

    def on_zoom_reset(self, event=None):
        if self.doc is not None:
            self.scroll_position /= (self.samples_per_frame / self.default_samples_per_frame)
            factor = (self.samples_per_frame / self.default_samples_per_frame)
            self.sample_width = self.default_sample_width
            self.samples_per_frame = self.default_samples_per_frame
            self.samples_per_sec = self.doc.fps * self.samples_per_frame
            self.frame_width = self.sample_width * self.samples_per_frame
            for node in self.main_node.descendants:
                node.name.after_reposition()
                node.name.fit_text_to_size()
            self.start_recalc()
            if self.temp_play_marker:
                self.temp_play_marker.setRect(self.temp_play_marker.rect().x(), 1, self.frame_width + 1, self.height())
            self.scene().setSceneRect(self.scene().sceneRect().x(), self.scene().sceneRect().y(),
                                      self.scene().sceneRect().width() / factor, self.scene().sceneRect().height())
            self.setSceneRect(self.scene().sceneRect())
            self.horizontalScrollBar().setValue(self.scroll_position)

# end of class WaveformView
