# -*- coding: utf-8 -*-

# tools.py
# This file is part of qarte-5
# Code name: Слава Україні! (Sláva Ukrayíni!)
# Author: Vincent Vande Vyvre <vincent.vandevyvre@oqapy.eu>
# Copyright: 2022-2025 Vincent Vande Vyvre
# Licence: GPL3
# Home page: https://launchpad.net/qarte

import os
import re
import string
import collections
import logging
lgg = logging.getLogger(__name__)

# Following code is taken from:
# https://github.com/jsonzilla/vtt-to-srt.py/blob/master/vtt_to_srt/vtt_to_srt.py
# Copyright: 2022 Jansen A. Simanullang / Jeison Cardoso
class SubtitleConverter:
    def __init__(self, vtt):
        self.vtt = vtt

    def vtt_to_srt(self):
        lgg.info("Convert vtt to srt")
        with open(self.vtt, "r") as orig:
            content = orig.read()
        new = self.convert_content(content)
        outf = self.vtt.replace(".vtt", ".srt")
        with open(outf, "w") as target:
            target.write(new)

    def convert_header(self, contents):
        """Convert of vtt header to srt format

       Args:
       contents -- str
       """
        rep = re.sub("WEBVTT\n", "", contents)
        rep = re.sub(r"Kind:[ \-\w]+\n", "", rep)
        rep = re.sub(r"Language:[ \-\w]+\n", "", rep)
        rep = re.sub(r"STYLE *\n::cue.*\n *color.*\n *}\n*", "", rep)
        rep = re.sub(r"STYLE *\n::cue.*\n *background-color.*\n *}\n*", "", rep)
        return rep

    def padding_timestamp(self, contents):
        """Add 00 to padding timestamp of to srt format

       Args:
       contents -- str
       """
        find_srt = string.Template(r'$a,$b --> $a,$b(?:[ \-\w]+:[\w\%\d:,.]+)*\n')
        minute = r"((?:\d\d:){1}\d\d)"
        second = r"((?:\d\d:){0}\d\d)"
        padding_minute = find_srt.substitute(a=minute, b=r"(\d{0,3})")
        padding_second = find_srt.substitute(a=second, b=r"(\d{0,3})")
        rep = re.sub(padding_minute, r"00:\1,\2 --> 00:\3,\4\n", contents)
        return re.sub(padding_second, r"00:00:\1,\2 --> 00:00:\3,\4\n", rep)

    def convert_timestamp(self, contents):
        """Convert timestamp of vtt file to srt format

        Args:
        contents -- str
        """
        find_vtt = string.Template(r'$a.$b --> $a.$b(?:[ \-\w]+:[\w\%\d:,.]+)*\n')
        all_stamp = find_vtt.substitute(a=r"((?:\d\d:){0,2}\d\d)", b=r"(\d{0,3})")
        return self.padding_timestamp(re.sub(all_stamp, 
                                        r"\1,\2 --> \3,\4\n", contents))

    def convert_content(self, contents):
        """Convert content of vtt file to srt format

        Args:
        contents -- str
        """
        rep = self.convert_timestamp(contents)
        rep = self.convert_header(rep)
        rep = re.sub(r"<c[.\w\d]*>", "", rep)
        rep = re.sub(r"</c>", "", rep)
        rep = re.sub(r"<\d\d:\d\d:\d\d.\d\d\d>", "", rep)
        rep = re.sub(r"::[\-\w]+\([\-.\w\d]+\)[ ]*{[.,:;\(\) \-\w\d]+\n }\n", 
                              "", rep)
        rep = re.sub(r"Style:\n##\n", "", rep)
        rep = self.add_sequence_numbers(rep)
        return rep
  
    def timestamp_line(self, content):
        """Check if line is a timestamp srt format

        Args:
        contents -- str
        """
        return re.match(r"((\d\d:){2}\d\d),(\d{3}) --> ((\d\d:){2}\d\d),(\d{3})", 
                        content) is not None

    def add_sequence_numbers(self, contents):
        """Adds sequence numbers to subtitle contents.

        Args:
        contents -- str
        """
        output = ''
        lines = contents.split(os.linesep)
        i = 1
        for line in lines:
            if self.timestamp_line(line):
                output += str(i) + os.linesep
                i += 1

            output += line + os.linesep

        return output


def disk_usage(path):
    """Return disk usage statistics about the given path.

    Returned value is a named tuple with attributes 'total', 'used' and
    'free', which are the amount of total, used and free space, in bytes.
    """
    _ntuple_diskusage = collections.namedtuple('usage', 'total used free')
    _ntuple_diskusage.total.__doc__ = 'Total space in bytes'
    _ntuple_diskusage.used.__doc__ = 'Used space in bytes'
    _ntuple_diskusage.free.__doc__ = 'Free space in bytes'
    st = os.statvfs(path)
    free = st.f_bavail * st.f_frsize
    total = st.f_blocks * st.f_frsize
    used = (st.f_blocks - st.f_bfree) * st.f_frsize
    return _ntuple_diskusage(total, used, free)




