/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * gui.c: Implementation, in the GUI toolkit in use, of the operations we need.
 */

#include "spettro.h"
#include "gui.h"

#include "audio.h"
#include "a_cache.h"
#include "axes.h"
#include "key.h"
#include "mouse.h"
#include "paint.h"	/* for do_scroll() */
#include "schedule.h"
#include "timer.h"
#include "ui.h"

/* Libraries' header files. See config.h for working combinations of defines */

#if HAVE_LIBPNG
#include <png.h>
#endif
#include <errno.h>

#if ECORE_TIMER || EVAS_VIDEO || ECORE_MAIN
#include <Ecore.h>
#include <Ecore_Evas.h>
#endif

#if EVAS_VIDEO
#include <Evas.h>
#endif

#if EMOTION_AUDIO
#include <Emotion.h>
#endif

#if SDL_AUDIO || SDL_TIMER || SDL_MAIN
# include <SDL.h>
#endif

/* Which video driver was selected during gui_init? */
static enum {
    video_none = 0,	/* No GUI is initialized */
#if EVAS_VIDEO
    video_efl,
#endif
#if SDL_VIDEO
    video_sdl,
#endif
#if DUMMY_VIDEO
    video_dummy,
#endif
} video_driver = video_none;

/* Internal data used to write on the image buffer */
#if EVAS_VIDEO
static Evas_Object *image;
static Ecore_Evas *ee;
static unsigned char *imagedata = NULL;
static size_t imagestride;		/* How many bytes per screen line ?*/
       Evas_Object *em = NULL;	/* The Emotion or Evas-Ecore object */
#endif

#if SDL_VIDEO
static SDL_Window *sdl_window;
static SDL_Renderer *sdl_renderer;
static SDL_Surface *sdl_surface = NULL; /* Only valid between calls to gui_{lock,unlock}() */
#endif

/* These are the EVAS values. SDL initializes them to different values. */
color_t gray	= 0xFF808080;
color_t green	= 0xFF00FF00;
color_t white	= 0xFFFFFFFF;
color_t black	= 0xFF000000;

#if SDL_VIDEO
/* When going into full-screen mode from a window, the display area's size
 * may change so we remember the non-full-screen window's width and height
 * and restore that size if we come back out of full-screen.
 */
static int windowed_width = -1, windowed_height = -1;
#endif

#if SDL_MAIN
static int get_next_SDL_event(SDL_Event *event);
#endif
#if ECORE_MAIN
static void quit_ecore_evas(Ecore_Evas *ee EINA_UNUSED);
#endif

/*
 * Initialize the GUI subsystem.
 *
 * "filename" is the name of the audio file, used for window title
 */
void
gui_init(char *filename)
{
    static char *last_filename = NULL;
#if EVAS_VIDEO
    Evas *canvas;
#endif
#if SDL_VIDEO
    SDL_Surface *surface;
#endif

    if (filename == NULL) filename = last_filename;
    else last_filename = filename;

    /*
     * Initialize the video subsystems.
     *
     * First try SDL which works best, then Enlightenment then libcaca.
     */

#if SDL_VIDEO
    if (SDL_InitSubSystem(SDL_INIT_VIDEO) != 0) {
	fprintf(stderr, "Couldn't initialize SDL video subsystem: %s.\n", SDL_GetError());
	goto end_sdl;
    }

    /* Use native resolution for fullscreen mode.
     * Letting it change video mode can both lock X up
     * and makes spettro get the screen update wrong.
     */
    if (fullscreen_mode) {
	SDL_DisplayMode mode;

	/* When going to fullscreen mode, remember the original size
	 * to be able to revert back to it if they go back to windowed mode. */
	windowed_width = disp_width;
	windowed_height = disp_height;
	if (SDL_GetDesktopDisplayMode(0, &mode) == 0) {
	    disp_width  = mode.w;
	    disp_height = mode.h;
	}
    } else {
	/* If they're coming out of fullscreen mode,
	 * go back to the original window size
	 */
	if (windowed_width >= 0 && windowed_height >= 0) {
	    disp_width = windowed_width;
	    disp_height = windowed_height;
	}
    }

    /* We create the window hidden, then show it when everything is done,
     * otherwise it flashes the window border, removes it and redisplays it */
    sdl_window = SDL_CreateWindow(filename,
			      SDL_WINDOWPOS_UNDEFINED, SDL_WINDOWPOS_UNDEFINED,
			      disp_width, disp_height,
			      SDL_WINDOW_HIDDEN |
			      (start_up_minimized ? SDL_WINDOW_MINIMIZED : 0) |
			      (fullscreen_mode ? SDL_WINDOW_FULLSCREEN : 0));
				 /* | SDL_WINDOW_RESIZABLE one day... */
    if (sdl_window == NULL) {
	fprintf(stderr, "Couldn't create SDL window: %s\n", SDL_GetError());
	exit(1);
    }

    sdl_renderer = SDL_CreateRenderer(sdl_window, -1, 0);
	/* maybe SDL_RENDERER_PRESENTVSYNC */
    if (sdl_renderer == NULL) {
	fprintf(stderr, "Couldn't create SDL renderer: %s\n", SDL_GetError());
	exit(1);
    }

    surface = SDL_GetWindowSurface(sdl_window);
    if (surface == NULL) {
	fprintf(stderr, "Couldn't get SDL Window Surface: %s\n",
		SDL_GetError());
	exit(1);
    }

    video_driver = video_sdl;	/* Need this here for RGB_to_color to work */

    gui_lock();
    background	= RGB_to_color(0x80, 0x80, 0x80);	/* 50% gray */
    green	= RGB_to_color(0x00, 0xFF, 0x00);
    white	= RGB_to_color(0xFF, 0xFF, 0xFF);
    black	= RGB_to_color(0x00, 0x00, 0x00);
    gui_unlock();

    /* Clear the image buffer to the background color */
    if (SDL_FillRect(surface, NULL, background) != 0) {
	fprintf(stderr, "Couldn't fill surface with background color: %s\n",
		SDL_GetError());
	exit(1);
    }

    SDL_ShowWindow(sdl_window);
    if (start_up_minimized) SDL_MinimizeWindow(sdl_window);
    goto got_video_driver;
end_sdl:
#endif

#if EVAS_VIDEO
    /* Initialize the graphics subsystem */
    if (!ecore_evas_init() ||
	!(ee = ecore_evas_new(NULL, 0, 0, disp_width, disp_height, NULL))) {
	fputs("Cannot initialize EFL graphics subsystem.\n", stderr);
	goto end_efl;
    }
    ecore_evas_callback_delete_request_set(ee, quit_ecore_evas);
    ecore_evas_title_set(ee, filename);
    if (fullscreen_mode) ecore_evas_fullscreen_set(ee, TRUE);
    ecore_evas_show(ee);

    canvas = ecore_evas_get(ee);

    /* Create the image and its memory buffer */
    image = evas_object_image_add(canvas);
    evas_object_image_colorspace_set(image, EVAS_COLORSPACE_ARGB8888);
    evas_object_image_size_set(image, disp_width, disp_height);
    imagestride = evas_object_image_stride_get(image);
    imagedata = Malloc(imagestride * disp_height);

    /* Clear the image buffer to the background color */
    {	register int i;
	register unsigned int *p = (unsigned int *)imagedata;

	for (i=(imagestride * disp_height) / sizeof(*p);
	     i > 0;
	     i--) {
	    *p++ = background;
	}
    }

    evas_object_image_data_set(image, imagedata);

    /* This gives an image that is automatically scaled with the window.
     * If you resize the window, the underlying image remains of the same size
     * and it is zoomed by the window system, giving a thick green line etc.
     */
    evas_object_image_filled_set(image, TRUE);
    ecore_evas_object_associate(ee, image, 0);

    evas_object_resize(image, disp_width, disp_height);
    evas_object_focus_set(image, EINA_TRUE); /* Without this no keydown events*/

    evas_object_show(image);

    /* Set GUI callbacks at a lower priority so that scroll events come first */
    evas_object_event_callback_priority_add(image, EVAS_CALLBACK_KEY_DOWN,
				   EVAS_CALLBACK_PRIORITY_AFTER, keyDown, em);
    evas_object_event_callback_priority_add(image, EVAS_CALLBACK_MOUSE_DOWN,
				   EVAS_CALLBACK_PRIORITY_AFTER, mouseDown, em);
    evas_object_event_callback_priority_add(image, EVAS_CALLBACK_MOUSE_UP,
				   EVAS_CALLBACK_PRIORITY_AFTER, mouseUp, em);
    evas_object_event_callback_priority_add(image, EVAS_CALLBACK_MOUSE_MOVE,
				   EVAS_CALLBACK_PRIORITY_AFTER, mouseMove, em);
    video_driver = video_efl;
    goto got_video_driver;
end_efl:
#endif

    fprintf(stderr, "No video driver managed to initialise.\n");
    exit(1);

got_video_driver:
    set_disp_offset();

    /* audio cache size depends on graph X dimension */
    reposition_audio_cache();

    /* Initialize the audio subsystem */
    switch (video_driver) {
#if EMOTION_AUDIO || EVAS_VIDEO
    case video_efl:
# if EMOTION_AUDIO
	em = emotion_object_add(canvas);
# elif EVAS_VIDEO
	em = evas_object_smart_add(canvas, NULL);
# endif
	if (!em) {
# if EMOTION_AUDIO
	    fputs("Couldn't initialize Emotion audio.\n", stderr);
# else
	    fputs("Couldn't initialize Evas graphics.\n", stderr);
# endif
	    exit(1);
	}
	break;
#endif
#if SDL_VIDEO
    case video_sdl:
	break;
#endif
    default:
	abort();
	break;
    }
}

/* Shut the graphics system down.
 * Done before quitting or (SDL) re-initting to change fullscreen mode.
 */

void
gui_deinit()
{
    switch (video_driver) {
#if EVAS_VIDEO
    case video_efl:
	ecore_evas_free(ee);
	/* This makes it dump core or barf error messages about bad magic */
	ecore_evas_shutdown();
	free(imagedata);
	break;
#endif

#if SDL_VIDEO
    case video_sdl:
	start_up_minimized = SDL_GetWindowFlags(sdl_window) & SDL_WINDOW_MINIMIZED;

	/* SDL_DestroyWindow rarely segfaults during exit.
	 * Workarounds, hoping to be able exec and play the next track. */
# if 0
	/* It could be that removing this makes a difference as
	 * "SDL_Quit deletes active windows and renderers"
	 * according to https://stackoverflow.com/questions/62977628
	 */
	SDL_DestroyRenderer(sdl_renderer);
# endif

	SDL_DestroyWindow(sdl_window);
	SDL_QuitSubSystem(SDL_INIT_VIDEO);
	/* SDL_Quit(); is called by atexit() if we are quitting */
	break;
#endif
    default:
	abort();
	break;
    }
}

/* Tell the video subsystem to update the display from the pixel data */
void
gui_update_display()
{
#if EVAS_VIDEO
    Evas *canvas = ecore_evas_get(ee);
#endif

    switch (video_driver) {
#if EVAS_VIDEO
    case video_efl:
	evas_object_image_data_update_add(image, 0, 0, disp_width, disp_height);
	evas_render_updates_free(evas_render_updates(canvas));
	break;
#endif
#if SDL_VIDEO
    case video_sdl:
	SDL_UpdateWindowSurface(sdl_window);
	break;
#endif
    default:
	abort();
	break;
    }
}

/* Tell the video subsystem to update a rectangle from the pixel data
 * The parameters are in our 0-at-bottom coordinates;
 * while Evas and SDL both have 0-at-top.
 */
void
gui_update_rect(int from_x, int from_y, int to_x, int to_y)
{
    int width = to_x - from_x + 1;
    int height = to_y - from_y + 1;
#if EVAS_VIDEO
    Evas *canvas;
#endif
#if SDL_VIDEO
    SDL_Rect rect;
#endif

    switch (video_driver) {
#if EVAS_VIDEO
    case video_efl:
	canvas = ecore_evas_get(ee);

	evas_object_image_data_update_add(image, from_x,
	    (disp_height - 1) - (from_y + height - 1), width, height);
	evas_render_updates_free(evas_render_updates(canvas));
	break;
#endif

#if SDL_VIDEO
    case video_sdl:
	rect.x = from_x;
	/* Our Y coordinates have their origin at the bottom, SDL at the top */
	rect.y = (disp_height - 1) - (from_y + height - 1);
	rect.w = width;
	rect.h = height;

	if (SDL_UpdateWindowSurfaceRects(sdl_window, &rect, 1) != 0) {
	    fprintf(stderr, "SDL_UpdateWindowSurfaceRects failed: %s\n",
		    SDL_GetError());
	}
	break;
#endif
    default:
	break;
    }
}

/* Tell the video subsystem to update one column of the display
 * from the pixel data
 */
void
gui_update_column(int pos_x)
{
    gui_update_rect(pos_x, min_y, pos_x, max_y);
}

void
gui_main()
{
#if SDL_MAIN
    SDL_Event event;
#endif

    switch (video_driver) {
#if ECORE_MAIN
    case video_efl:
	/* Start main event loop */
	ecore_main_loop_begin();
	break;
#endif

#if SDL_MAIN
    case video_sdl:
	/* Use SDL2's TEXTINPUT mode so that keyboard mapping with Shift and
	 * AltGr works */
	SDL_StartTextInput();

	while (get_next_SDL_event(&event)) switch (event.type) {
	case SDL_WINDOWEVENT:
	    switch (event.window.event) {
	    case SDL_WINDOWEVENT_EXPOSED:
		gui_update_display();
		break;
	    case SDL_WINDOWEVENT_MINIMIZED:
		start_up_minimized = TRUE;
		break;
	    case SDL_WINDOWEVENT_RESTORED:
		start_up_minimized = FALSE;
		break;
	    }
	    break;

	case SDL_QUIT:
	    /* This happens when gui_quit_main_loop() is called or
	     * they've clicked the desktop border's [X] icon or hit ALT-F4.
	     * When gui_quit_main_loop() is called, one of quitting and nexting
	     * will be true. If it's a desktop close, both are false so force a
	     * full quit.
	     */
	    if (!quitting && !nexting) quitting = TRUE;
	    return;

	/* For SDL2, we enable both KEYDOWN and TEXTINPUT because
	 * TEXTINPUT handles Shift and AltGr to get difficult chars on
	 * international keyboards, but ignores arrow keys and the keypad.
	 * in key.c, if SDL2, we process most keys with TEXTINPUT and only
	 * the ones TEXTINPUT ignores in response to KEYDOWN.
	 */

	case SDL_KEYDOWN:
	    /* SDL's event.key.keysym.mod reflects the state of the modifiers
	     * at initial key-down. SDL_GetModState seems to reflect now. */
	    Shift = !!(SDL_GetModState() & KMOD_SHIFT);
	    Ctrl  = !!(SDL_GetModState() & KMOD_CTRL);
	    sdl_keydown(&event);
	    break;

	case SDL_TEXTINPUT:
	    Shift = !!(SDL_GetModState() & KMOD_SHIFT);
	    Ctrl  = !!(SDL_GetModState() & KMOD_CTRL);
	    sdl_keydown(&event);
	    break;

	case SDL_MOUSEBUTTONDOWN:
	case SDL_MOUSEBUTTONUP:
	    {
		/* To detect Shift and Ctrl states, it looks like we have to
		 * examine the keys ourselves */
		SDL_Keymod state = SDL_GetModState();

		Shift = !!(state & KMOD_SHIFT);
		Ctrl = !!(state & KMOD_CTRL);

		switch (event.button.button) {
		case SDL_BUTTON_LEFT:
		case SDL_BUTTON_RIGHT:
		    do_mouse_button(event.button.x, event.button.y,
				    event.button.button == SDL_BUTTON_LEFT
				    ? LEFT_BUTTON : RIGHT_BUTTON,
				    event.type == SDL_MOUSEBUTTONDOWN
				    ? MOUSE_DOWN : MOUSE_UP);
		}
	    }
	    break;

	case SDL_MOUSEMOTION:
	    do_mouse_move(event.motion.x, event.motion.y);
	    break;

	case SDL_USEREVENT:
	    switch (event.user.code) {
	    case RESULT_EVENT:
		/* Column result from a calculation thread */
		calc_notify((calc_t *) event.user.data1);
		break;
	    case SCROLL_EVENT:
		do_scroll();
		break;

	    default:
		fprintf(stderr, "Unknown SDL_USEREVENT code %d\n",
			event.user.code);
		break;
	    }
	    break;

	default:
	    break;
	}
	break;
#endif
    default:
	abort();
	break;
    }
}

#if SDL_MAIN
static int
get_next_SDL_event(SDL_Event *eventp)
{
    /* Prioritize UI events over window refreshes, results and such */
    /* First, see if there are any UI events to be had */
    SDL_PumpEvents();

    /* First priority: Quit */
    if (SDL_PeepEvents(eventp, 1, SDL_GETEVENT, SDL_QUIT, SDL_QUIT) == 1)
	return 1;

    /* Second priority: UI events
     *
     * SDL_{KEYDOWN,KEYUP,TEXEDITING,TEXTINPUT} are consecutive and followed by
     * SDL_MOUSE{MOTION,BUTTONDOWN,BUTTONUP,WHEEL}.
     * see /usr/include/SDL2/SDL_events.h
     */
    while (SDL_PeepEvents(eventp, 1, SDL_GETEVENT,
				     SDL_KEYDOWN, SDL_MOUSEWHEEL) == 1) {
	switch (eventp->type) {
	case SDL_KEYDOWN:
	case SDL_TEXTINPUT:
	case SDL_MOUSEBUTTONDOWN:
	case SDL_MOUSEBUTTONUP:
	    return 1;
	case SDL_MOUSEMOTION:
	    /* Only action the last mousemove event, to avoid redrawing bar lines
	     * for every pixel move, which gets slow when the barlines are close.
	     */
	    do {} while (SDL_PeepEvents(eventp, 1, SDL_GETEVENT,
					SDL_MOUSEMOTION, SDL_MOUSEMOTION) == 1);
	    return 1;
	default:
	    /* Ignore keyboard/mouse events that are not known to gui_main() */
	    continue;
	}
    }

    /* Third priority: screen-scrolling events */
    if (SDL_PeepEvents(eventp, 1, SDL_GETEVENT, SDL_USEREVENT, SDL_USEREVENT) == 1)
	return 1;

    /* No Quit or UI events? Wait for all events */
    return SDL_WaitEvent(eventp);
}
#endif

#if ECORE_MAIN
/* Ecore callback for the same */
static void
quit_ecore_evas(Ecore_Evas *ee EINA_UNUSED)
{
    gui_quit_main_loop();
}
#endif

void
gui_quit_main_loop(void)
{
#if SDL_MAIN
    SDL_Event event;
#endif

    switch (video_driver) {
#if ECORE_MAIN
    case video_efl:
	ecore_main_loop_quit();
	break;
#endif
#if SDL_MAIN
    case video_sdl:
    event.type = SDL_QUIT;
	if (SDL_PushEvent(&event) != SDL_PUSHEVENT_SUCCESS) {
	    fprintf(stderr, "sdl_main_loop_quit event push failed: %s\n", SDL_GetError());
	    exit(1);
	}
	break;
#endif
    default:
	abort();
	break;
    }
}

/*
 * Flip between windowed and full-screen mode in response to the Ctrl-F key.
 * "fullscreen" in ui.c knows whether we are in fullscreen mode already or not.
 */
void
gui_fullscreen()
{
    switch (video_driver) {
#if EVAS_VIDEO
    case video_efl:
	fullscreen_mode = !fullscreen_mode;
	ecore_evas_fullscreen_set(ee, fullscreen_mode);
	/* Evas zooms the window at the same resolution to screen size.
	 * When it will be able to change the display size,
	 * remember to set_disp_offset(). */
	break;
#endif
#if SDL_VIDEO
    case video_sdl:
	gui_deinit();
	fullscreen_mode = !fullscreen_mode;
	gui_init(NULL);	/* Reuse the same file name as title */
	draw_axes();
	repaint_display(FALSE);
	break;
#endif
    default:
	abort();
	break;
    }
}

void
gui_h_scroll_by(int scroll_by)
{
#if EVAS_VIDEO
    Evas *canvas = ecore_evas_get(ee);
    int width = max_x - min_x + 1;
    int height = max_y - min_y + 1;
    int y;
#endif
#if SDL_VIDEO
    SDL_Rect from, to;
    int err;
#endif

    switch (video_driver) {
#if EVAS_VIDEO
    case video_efl:
	if (scroll_by > 0) {
	    /* Usual case: scrolling the display left to advance in time */
	    for (y=min_y; y <= max_y; y++) {
		memmove(imagedata+y*imagestride + (4 * min_x),
			imagedata+y*imagestride + (4 * min_x) + (4 * scroll_by),
			4 * width - 4 * scroll_by);
	    }
	}
	if (scroll_by < 0) {
	    /* Scrolling back in time */
	    for (y=min_y; y <= max_y; y++)
		memmove(imagedata+y*imagestride + (4 * min_x) + (4 * -scroll_by),
			imagedata+y*imagestride + (4 * min_x),
			4 * width - (4 * -scroll_by));
	}
	if (scroll_by != 0) {
	    evas_object_image_data_update_add(image, min_x, min_y, width, height);
	    evas_render_updates_free(evas_render_updates(canvas));
	}
	break;
#endif
#if SDL_VIDEO
    case video_sdl:
	if (scroll_by > 0) {
	    /* Usual case: scrolling the display left to advance in time */
	    from.x = min_x + scroll_by;
	    to.x = min_x;
	    from.w = max_x - min_x + 1 - scroll_by;
	    /* to.[wh] are ignored */
	}
	if (scroll_by < 0) {
	    /* Happens when they seek back in time */
	    from.x = min_x;
	    to.x = min_x -scroll_by;
	    from.w = max_x - min_x + 1 - -scroll_by;
	    /* to.[wh] are ignored */
	}
	if (scroll_by != 0) {
	    SDL_Surface *surface = SDL_GetWindowSurface(sdl_window);
	    if (surface == NULL) {
		fprintf(stderr, "Couldn't get SDL Window Surface: %s\n",
			SDL_GetError());
		exit(1);
	    }
	    from.y = to.y = (disp_height - 1) - max_y;
	    from.h = max_y - min_y + 1;
	    if ((err = SDL_BlitSurface(surface, &from, surface, &to)) != 0) {
		fprintf(stderr, "SDL Blit failed with value %d.\n", err);
	    }
	}
	break;
#endif
    default:
	abort();
	break;
    }
}

/* Scroll the graph vertically by a number of pixels.
 * A positive value of scroll_by means to move to higher frequencies by
 * moving the graphic data downwards; a negative value to lower frequencies
 * by moving the displayed data upward.
 */
void
gui_v_scroll_by(int scroll_by)
{
    int width = max_x - min_x + 1;
    int height = max_y - min_y + 1;
#if EVAS_VIDEO
    int y;	/* destination y coordinate */
#endif
#if SDL_VIDEO
    SDL_Rect from, to;
    int err;
#endif

    switch (video_driver) {
#if EVAS_VIDEO
    case video_efl:
	if (scroll_by > 0) {
	    /* Move to higher frequencies by scrolling the graphic down.
	     * Copy lines downwards, i.e. forwards in memory and
	     * start from the bottom and work upwards */
	    for (y=min_y; y <= max_y-scroll_by; y++)
		memmove(imagedata + 4*min_x + imagestride*(disp_height-1-y),
			imagedata + 4*min_x + imagestride*(disp_height-1-y-scroll_by),
			4*width);
	}
	if (scroll_by < 0) {
	    /* Move to lower frequencies by scrolling the graphic up.
	     * Copy lines upwards, i.e. to lower memory */
	    for (y=max_y; y >= min_y-scroll_by; y--)
		memmove(imagedata + 4*min_x + imagestride*(disp_height-1-y),
			imagedata + 4*min_x + imagestride*(disp_height-1-y-scroll_by),
			4*width);
	}
	if (scroll_by != 0) {
	    Evas *canvas = ecore_evas_get(ee);

	    evas_object_image_data_update_add(image, min_x, min_y, width, height);
	    evas_render_updates_free(evas_render_updates(canvas));
	}
	break;
#endif

#if SDL_VIDEO
    case video_sdl:
	if (scroll_by > 0) {
	    /* Move to higher frequencies by scrolling the graphic down */
	    /* Invert y coords because SDL is 0-at-top and we have 0 at bottom */
	    from.x = min_x;
	    from.y = (disp_height - 1) - max_y;
	    from.w = width;
	    from.h = height - scroll_by;
	    to.x = min_x;
	    to.y = (disp_height - 1) - (max_y - scroll_by);
	    /* to.[wh] are ignored */
	}
	if (scroll_by < 0) {
	    /* Move to lower frequencies by scrolling the graphic up */
	    from.x = min_x;
	    from.y = (disp_height - 1) - max_y - scroll_by;
	    from.w = width;
	    from.h = height - -scroll_by;
	    to.x = min_x;
	    to.y = (disp_height - 1) - max_y;
	    /* to.[wh] are ignored */
	}
	if (scroll_by != 0) {
	    SDL_Surface *surface = SDL_GetWindowSurface(sdl_window);
	    if (surface == NULL) {
		fprintf(stderr, "Couldn't get SDL Window Surface: %s\n",
			SDL_GetError());
		exit(1);
	    }
	    if ((err = SDL_BlitSurface(surface, &from, surface, &to)) != 0) {
		fprintf(stderr, "SDL Blit failed with value %d.\n", err);
	    }
	}
	break;
#endif
    default:
	abort();
	break;
    }
}

/* Fill a rectangle with a single colour" */
void
gui_paint_rect(int from_x, int from_y, int to_x, int to_y, color_t color)
{
#if EVAS_VIDEO
    unsigned char *p;	/* pointer to pixel to set */
    int y, x;
#endif
#if SDL_VIDEO
    SDL_Surface *surface;
#endif

    switch (video_driver) {
#if EVAS_VIDEO
    case video_efl:
	/* Paint top to bottom so that we move forward in the imagedata */
	for (y=to_y,
	     p = (unsigned char *)((unsigned int *)imagedata)
				    + (disp_height-1-to_y) * imagestride;
	     y >= from_y;
	     y--, p += imagestride)
		for (x=from_x; x <= to_x; x++)
		    ((unsigned int *)p)[x] = color;
	break;
#endif

#if SDL_VIDEO
    case video_sdl:
	surface = SDL_GetWindowSurface(sdl_window);
	if (surface == NULL) {
	    fprintf(stderr, "Couldn't get SDL Window Surface: %s\n",
		    SDL_GetError());
	    exit(1);
	}
	SDL_Rect rect = {
	    from_x, (disp_height-1)-to_y, /* SDL is 0-at-top, we are 0-at-bottom */
	    to_x - from_x + 1, to_y - from_y + 1
	};
	SDL_FillRect(surface, &rect, color);
	break;
#endif
    default:
	abort();
	break;
    }
}

/* Fill a pixel column with a single colour, probably "green" or "background" */
void
gui_paint_column(int pos_x, int from_y, int to_y, unsigned int color)
{
    /* gui_paint_column is only called for the display area */
    if (pos_x >= min_x && pos_x <= max_x)
	gui_paint_rect(pos_x, from_y, pos_x, to_y, color);
}

/* Draw a horizontal or vertical line.
 * The caller should put us inside a gui_lock/unlock() pair.
 */
void
gui_draw_line(int x1, int y1, int x2, int y2, color_t color)
{
    if (x1 > x2) {
	int tmp = x1; x1 = x2; x2 = tmp;
    }
    if (y1 > y2) {
	int tmp = y1; y1 = y2; y2 = tmp;
    }
    /* SDL2's SDL_RenderDrawLine() makes the whole display flicker */
    /* but we only have to do horizontal and vertical lines */
    if (x1 == x2) {
	int y;
	for (y=y1; y <= y2; y++)
	    gui_putpixel(x1, y, color);
    } else if (y1 == y2) {
	int x;
	for (x=x1; x <= x2; x++)
	    gui_putpixel(x, y1, color);
    }
}

/* These must be put around calls to gui_putpixel() and RGB_to_color() */
void
gui_lock()
{
    switch (video_driver) {
#if SDL_VIDEO
    case video_sdl:
	sdl_surface = SDL_GetWindowSurface(sdl_window);
	if (sdl_surface == NULL) {
	    fprintf(stderr, "Couldn't get SDL Window Surface: %s\n",
		    SDL_GetError());
	    exit(1);
	}
	if (SDL_MUSTLOCK(sdl_surface) && SDL_LockSurface(sdl_surface) != 0 ) {
	    fprintf(stderr, "Can't lock surface: %s\n", SDL_GetError());
	}
	break;
#endif
    default:
	break;
    }
}

void
gui_unlock()
{
    switch (video_driver) {
#if SDL_VIDEO
    case video_sdl:
	if (SDL_MUSTLOCK(sdl_surface)) SDL_UnlockSurface(sdl_surface);
	sdl_surface = NULL;
	break;
#endif
    default:
	break;
    }
}

/* Convert a 0-255 value of red, green and blue to a color_t.
 *
 * Calls to this must be wrapped with gui_{lock,unlock}() so that
 * sdl_surface is valid.
 */
color_t
RGB_to_color(primary_t red, primary_t green, primary_t blue)
{
    switch (video_driver) {
#if EVAS_VIDEO
    case video_efl:
	return blue | (green << 8) | (red << 16) | 0xFF000000;
#endif

#if SDL_VIDEO
    case video_sdl:
	if (sdl_surface == NULL) {
	    /* Has happened once, in call paint_column->colormap->RGB_to_color */
	    fprintf(stderr, "Internal error: Calls to RGB_to_color() must be wrapped with gui_lock()/gui_unlock()\n");
	    /* Mystery. Try anyway. */
	    sdl_surface = SDL_GetWindowSurface(sdl_window);
	    if (sdl_surface == NULL) abort();
	}
	return SDL_MapRGB(sdl_surface->format, red, green, blue);
#endif
    default:
	abort();
	break;
    }
}

/* Calls to this should be bracketed by gui_lock() and gui_unlock(). */
void
gui_putpixel(int x, int y, color_t color)
{
#if EVAS_VIDEO
    color_t *row;	/* of pixels */
#endif

    if (x < 0 || x >= disp_width ||
	y < 0 || y >= disp_height) return;

    switch (video_driver) {
#if EVAS_VIDEO
    case video_efl:
	row = (color_t *)&imagedata[imagestride * ((disp_height-1) - y)];
	row[x] = color;
	break;
#endif
#if SDL_VIDEO
    case video_sdl:
	if (sdl_surface == NULL) {
	    fprintf(stderr, "Internal error: Calls to gui_putpixel() must be wrapped with gui_lock()/gui_unlock()\n");
	    abort();
	}
    /* Macro derived from http://sdl.beuc.net/sdl.wiki/Pixel_Access's putpixel() */
    #define putpixel(x, y, pixel) \
	    ((Uint32 *)((Uint8 *)sdl_surface->pixels + (y) * sdl_surface->pitch))[x] = pixel

	/* SDL has y=0 at top */
	putpixel(x, (disp_height-1) - y, color);
	break;
#endif
    default:
	abort();
	break;
    }
}

/* Dump the current screen contents to a PNG file.
 *
 * Returns TRUE on success, FALSE on failure
 */
bool
gui_output_png_file(const char *filename)
{
#if !HAVE_LIBPNG
    fprintf(stderr, "Screen dumping requires compilation with libpng.\n");
    return FALSE;
#else
    png_structp png_ptr;
    png_infop   png_info;
    png_bytepp  png_rows;
    png_uint_32 png_width = disp_width;
    png_uint_32 png_height = disp_height;
    FILE *file = NULL;
    int y;
# if SDL_VIDEO
    SDL_Surface *surface;
# endif

    png_ptr  = png_create_write_struct(PNG_LIBPNG_VER_STRING, 0, 0, 0);
    if (png_ptr == NULL) {
	fprintf(stderr, "Can't create PNG write structure.\n");
	return FALSE;
    }
    if (png_width > png_get_user_width_max(png_ptr) ||
	png_height > png_get_user_height_max(png_ptr)) {
	fprintf(stderr, "Image is too large for libpng.\n");
	png_destroy_write_struct(&png_ptr, (png_infopp)NULL);
	return FALSE;
    }

    png_info = png_create_info_struct(png_ptr);
    if (png_info == NULL) {
	fprintf(stderr, "Can't create PNG info structure.\n");
	png_destroy_write_struct(&png_ptr, (png_infopp)NULL);
	return FALSE;
    }

    /* Set up handling of errors within libpng */
    if (setjmp(png_jmpbuf(png_ptr))) {
	png_destroy_write_struct(&png_ptr, &png_info);
	if (file != NULL) fclose(file);
	return FALSE;
    }

    png_rows = png_malloc(png_ptr, png_height * sizeof (png_bytep));

    green_line_off = TRUE;
    repaint_display(TRUE);	/* repainting all reflects a changed logmax */

    file = fopen(filename, "wb");
    if (file == NULL) {
	fprintf(stderr, "Can't open \"%s\": %s\n", filename, strerror(errno));
	png_destroy_write_struct(&png_ptr, &png_info);
	return FALSE;
    }

    png_init_io(png_ptr, file);
    png_set_IHDR(png_ptr, png_info, png_width, png_height,
		 8, PNG_COLOR_TYPE_RGBA, PNG_INTERLACE_ADAM7,
		 PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

    switch (video_driver) {
# if SDL_VIDEO
    case video_sdl:
	surface = SDL_GetWindowSurface(sdl_window);
	if (surface == NULL) {
	    fprintf(stderr, "Couldn't get SDL Window Surface: %s\n",
		    SDL_GetError());
	    exit(1);
	}
	for (y = 0; y < png_height; y++)
	    png_rows[y] = (png_bytep)
		((Uint8 *)surface->pixels + y * surface->pitch);
	break;
# endif

# if EVAS_VIDEO
    case video_efl:
	for (y = 0; y < png_height; y++)
	    png_rows[y] = (png_bytep)
		(&imagedata[imagestride * y]);
	break;
# endif
    default:
	abort();
	break;
    }

    png_set_rows(png_ptr, png_info, png_rows);
    png_write_png(png_ptr, png_info,
# if SDL_VIDEO
		  ((video_driver == video_sdl) ? PNG_TRANSFORM_INVERT_ALPHA
					      : 0) |
# endif
		  PNG_TRANSFORM_BGR,
		  NULL);
    png_write_end(png_ptr, png_info);
    png_free(png_ptr, png_rows);
    png_destroy_write_struct(&png_ptr, &png_info);
    fclose(file);

    /* If just outputting a PNG and quitting (-o), no need to restore display */
    if (!output_file) {
	green_line_off = FALSE;
	repaint_column(disp_offset, min_y, max_y, FALSE);
	gui_update_column(disp_offset);
	printf("Dumped the window to %s\n", filename);
    }

    return TRUE;
#endif
}
