/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* mpg123.c: Interface between spettro and libmpg123 for MP3 decoding
 *
 * Adapted from mpg123's example program feedseek.c, which is copyright 2008
 * by the mpg123 project - free software under the terms of the LGPL 2.1
 */

#include "spettro.h"
#include "mpg_123.h"

#if HAVE_LIBMPG123

#include <string.h>	/* for strerror() */
#include <errno.h>
#include <mpg123.h>
#if HAVE_LIBMAGIC
# include <magic.h>
#endif

/* Open an MP3 file, setting af->{sample_rate,channels,frames}
 * On failure, returns FALSE.
 */
bool
libmpg123_open(audio_file_t *af, char *filename)
{
    int ret;
    static bool initialized = FALSE;

    /* Unfortunately, libmpg123_{open,getformat}() "succeeds" in opening
     * some WAV files and some Ogg files but returns the file length as
     * 0.048 seconds and gives garbage data, so do a pre-emptive check
     * on the file type.
     */
#if HAVE_LIBMAGIC
    magic_t magic;
    const char *mime_type;

    magic = magic_open(MAGIC_SYMLINK | MAGIC_MIME_TYPE | MAGIC_ERROR);
    if (magic == NULL) {
	fprintf(stderr, "Error opening libmagic");
	if (magic_error(magic) != NULL)
	    fprintf(stderr, ": %s\n", magic_error(magic));
	else
	    perror("");
	return FALSE;
    }
    if (magic_load(magic, NULL) != 0 ||
        (mime_type = magic_file(magic, filename)) == NULL ||
	strcmp(mime_type, "audio/mpeg") != 0) {
	magic_close(magic);
	return FALSE;
    }

    magic_close(magic);
#else
# warning "Using libmpg123 without libmagic opens some Ogg/WAV files with libmpg123 and returns garbage"
#endif
 
    if (!initialized) {
	mpg123_init();
	initialized = TRUE;
    }

    if ((af->mh = mpg123_new(NULL, &ret)) == NULL) {
	fprintf(stderr,"Unable to create mpg123 handle: %s\n",
		mpg123_plain_strerror(ret));
	return FALSE;
    }

    /* Suppress warning messages from libmpg123 */
    mpg123_param(af->mh, MPG123_ADD_FLAGS, MPG123_QUIET, 0.0);

    if (mpg123_open(af->mh, filename) != MPG123_OK) goto fail_delete;

    /* Set the output format to signed short but keep the sampling rate */
    {
	long rate; int channels; int encoding;

	if (mpg123_getformat(af->mh, &rate, &channels, &encoding) != MPG123_OK)
	    goto fail_close;

	/* Use 16-bit signed output, right for the cache */
	if (mpg123_format_none(af->mh) != MPG123_OK ||
	    mpg123_format(af->mh, rate,  MPG123_MONO | MPG123_STEREO,
					  MPG123_ENC_SIGNED_16) != MPG123_OK)
	    goto fail_close;

	af->channels = channels;
	af->sample_rate = rate;

	/* Remember these in copies that may change mid-stream */
	af->mpg123_rate = rate;
	af->mpg123_channels = channels;
	af->mpg123_encoding = encoding;

	/* We may need to call mpg123_scan() to determine the track length
	 * accurately, but that takes ten seconds for an hour-long track
	 * due to disk IO time. However,
	 *
	 > On 23/08/2020, Thomas Orgis <thomas-forum@orgis.org> wrote:
	 > 1. Call mpg123_info() and check for the vbr mode.
	 > If it's not MPG123_CBR, you had some Xing/Lame/Info frame
	 > to tell the decoder that. Track length is a common feature
	 > that all these frames have.
	 >
	 > 2. Call mpg123_getstate() for MPG123_ENC_DELAY and/or
	 > MPG123_ENC_PADDING. If one of them is != -1, you got a
	 > Lame info tag that contained such a value, also implying
	 > that there is proper length info.
	 *
	 * So we do that.
	 */
	{
	    struct mpg123_frameinfo mi;
#if MPG123_API_VERSION >= 45
	    long val;
#endif

	    if (!((mpg123_info(af->mh, &mi) == MPG123_OK &&
		   mi.vbr != MPG123_CBR)
/* These are only present from libmpg123 1.26 */
#if MPG123_API_VERSION >= 45
		   ||
		  (mpg123_getstate(af->mh, MPG123_ENC_DELAY, &val, NULL)
		   == MPG123_OK && val != -1)
		   ||
		  (mpg123_getstate(af->mh, MPG123_ENC_PADDING, &val, NULL)
		   == MPG123_OK && val != -1)
#endif
		   )) (void) mpg123_scan(af->mh);
	}
	{
	    frames_t length = mpg123_length(af->mh);
	    if (length < 0) {
		fprintf(stderr, "Can't discover the length of the MP3 file.\n");
		goto fail_close;
	    }
	    af->frames = length;
	}
    }

    af->filename = filename;

    return TRUE;

fail_close:
    mpg123_close(af->mh);
fail_delete:
    mpg123_delete(af->mh);
    af->mh = NULL;
    return FALSE;
}

/* Seek to the "start"th sample from the MP3 file.
 * Returns TRUE on success, FALSE on failure.
 */
bool
libmpg123_seek(audio_file_t *af, int start)
{
    off_t err;

    if ((err = mpg123_seek(af->mh, (frames_t)start, SEEK_SET)) < 0) {
	fprintf(stderr, "Failed to seek in MP3 file: %s\n", mpg123_plain_strerror(err));
	return FALSE;
    }

    return TRUE;
}

/*
 * Read samples from the MP3 file and convert them to the desired format
 * into the buffer "write_to".
 *
 * Returns the number of frames written, or a negative value on errors.
 */
int
libmpg123_read_frames(audio_file_t *af,
		      void *write_to,
		      frames_t frames_to_read,
		      af_format_t format)
{
    /* Number of sample frames written into write_to */
    int frames_written = 0; /* Quieten "may be used uninitialized" warning */
    off_t r;

    /* Avoid reading past end of file because that makes the whole read fail */
    r = mpg123_tell(af->mh);
    if (r < 0) {
	fprintf(stderr, "libmpg123_read_frames(): libmpg123 can't tell where it is!\n");
	return -1;
    }

    /* mpg123_tell() returns result in samples, not sample frames. */
    r /= af->mpg123_channels;

    if (r + frames_to_read > af->frames) {
	fprintf(stderr, "mpg123_tell returned %ld;\n", r);
	fprintf(stderr, "Reducing read length from %ld ", frames_to_read);
	frames_to_read = af->frames - r;
	fprintf(stderr, "to %ld\n", frames_to_read);
    }

    switch (format) {
	size_t bytes_to_read, bytes_written;
	int err;
    case af_signed:
	if (af->channels == af->mpg123_channels)
	    bytes_to_read = frames_to_read * sizeof(short) * af->channels;
	else if (af->channels == 2 && af->mpg123_channels == 1)
	    /* A blip of mono in a stereo file */
	    bytes_to_read = frames_to_read * sizeof(short);
	else if (af->channels == 1 && af->mpg123_channels == 2)
	    /* A blip of stereo in a mono file. Instead of allocating twice
	     * the memory and monoising it, read into the given buffer then
	     * monoise it and say we only got half the samples requested.
	     * They'll call back. */
	    bytes_to_read = (frames_to_read /= 2) * sizeof(short) * 2;
	else
	    /* Cannot happen but shut the compiler up about used-uninitialzed */
	    bytes_to_read = 0;

	bytes_written = 0; /* In case on error it doesn't write it (it does) */

	err = mpg123_read(af->mh, write_to, bytes_to_read, &bytes_written);
	if (af->channels == af->mpg123_channels)
	    frames_written = bytes_written / (sizeof(short) * af->channels);
	else if (af->channels == 2 && af->mpg123_channels == 1) {
	    /* The only case we've seen, a blip of mono in a stereo file */
	    int i;
	    short *ip, *op;

	    /* Stereoify the mono samples, working from the end to the start
	     * so as to read the data before they are overwritten */
	    for (i=(bytes_written / sizeof(short)) - 1,
		 ip = &(((short *)write_to)[i]),
		 op = &(((short *)write_to)[2*i]); i >= 0; i--) {
		op[0] = op[1] = *ip--;
		op -= 2;	/* --op--, anyone? */
	    }
	    frames_written = bytes_written / sizeof(short);
	    fprintf(stderr, "Stereoified %d frames\n", frames_written);
	} else if (af->channels == 1 && af->mpg123_channels == 2) {
	    /* A blip of stereo in a mono file */
	    int i;
	    short *ip = (short *)write_to;
	    short *op = (short *)write_to;

	    /* Monoise the stereo samples, working from start to end */
	    frames_written = bytes_written / (sizeof(short) * 2);
	    for (i=0; i < frames_written; i++) {
		*op++ = (ip[0] + ip[1]) / 2;
		ip += 2;
	    }
	}

	switch (err) {
	case MPG123_OK:
	    break;
	case MPG123_ERR:
	    return -1;
	case MPG123_DONE:
	    /* Happens with some broken MP3 files, e.g. Riders On The Storm,
	     * which then gives NEW_FORMAT and can continue. */
	    return -1;
	case MPG123_NEW_FORMAT:
	    /* "Output format will be different on next call" */
	    {
		long rate;
		int encoding;
		int channels;

		err = mpg123_getformat(af->mh, &rate, &channels, &encoding);
		switch (err) {
		case MPG123_OK:
		    break;
		default:
		    fprintf(stderr, "mpg123_getformat() fails: %s\n", mpg123_plain_strerror(err));
		    return -1;
		}

		if (rate != af->mpg123_rate) {
		    /* We complain but don't do anything */
		    fprintf(stderr, "mpg123's sample rate changed mid-stream from %ld to %ld\n",
			    af->mpg123_rate, rate);
		    af->mpg123_rate = rate;
		}

		/* Changes in the number of channels are handled above */
		af->mpg123_channels = channels;

		if (encoding != af->mpg123_encoding) {
		    /* We complain but don't do anything */
		    fprintf(stderr, "mpg123's encoding changed mid-stream from %d to %d\n",
			    af->mpg123_encoding, encoding);
		    af->mpg123_encoding = encoding;
		}
	    }
	    break;
	default:
	    return -1;
	}

	break;

    case af_float:  
	{
	    signed short *buf;

	    buf = Malloc(frames_to_read * af->channels * sizeof(*buf));

	    /* Reuse the above code to get format changes right */
	    frames_written = libmpg123_read_frames(af, (void *)buf,
						   frames_to_read, af_signed);
	    if (frames_written < 0) {
		free(buf);
		return -1;
	    }

	    /* MPEG files can only be mono or stereo */
	    switch (af->channels) {
		float *fp; signed short *sp; int i;
	    case 1:
		sp = buf; fp = write_to;
		for (i=0; i<frames_written; i++)
		    *fp++ = *sp++ / (float)32768;
		break;
	    case 2:
		sp = buf; fp = write_to;
		for (i=0; i<frames_written; i++) {
		    *fp++ = (sp[0] + sp[1]) / (float)65536;
		    sp += 2;
		}
		break;
	    }
	    free(buf);
	}
	break;
    }

    return frames_written;
}

void
libmpg123_close(audio_file_t *af)
{
    mpg123_close(af->mh);
    mpg123_delete(af->mh);
}
#endif /* HAVE_LIBMPG123 */
