/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * r_cache.c - Caching of FFT results
 *
 * We remember the FFT results we have calculated as calc_t's, in time order,
 * which reflect the parameters that gave rise to that result and the
 * linear magnitude data.
 *
 * The mapping from this to screen coordinates and colors is done on-the-fly
 * at screen update time by paint_column() calling interpolate() and colormap().
 *
 * This file is only called from the main thread, so no locking is required.
 */

#include "spettro.h"
#include "r_cache.h"

#include "convert.h"
#include "calc.h"
#include "window.h"
#include "ui.h"

#include <string.h>	/* for memcmp() */

static void destroy_result(calc_t *r);

static calc_t *results = NULL; /* Linked list of result structures */
static calc_t *last_result = NULL; /* Last element in the linked list */

/* Remember an FFT result that just arrived from a calculation thread.
 *
 * We return the result because, if we find a duplicate in the cache,
 * we free the new result and the existing one becomes the active result.
 */
calc_t *
remember_result(calc_t *result)
{
    frames_t earliest;	/* Drop all results before this time */
    frames_t latest;	/* Drop all results after this time */
    calc_t *r;		/* Loop pointer to run down the list */

    /* Drop stored results more than a screenful before the display */
    earliest = (frames_t) round(screen_column_to_start_time(min_x - disp_width) * sr);
    while (results != NULL && results->t < earliest) {
	r = results;  /* Remember the cell to drop so we can free it */

	/* Drop the first result in the cache */
	results = results->next;
	if (results == NULL) {
	    /* We have freed all the results */
	    last_result = NULL;
	    break;
	} else {
	    results->prev = NULL;
	}
	destroy_result(r);
    }

    /* Drop stored results more than a screenful after the display */
    latest = round(screen_column_to_start_time(max_x + disp_width) * sr);
    while (last_result != NULL && last_result->t > latest) {
	r = last_result;  /* Remember the cell to drop, to free it */

	/* Drop the last result in the cache */
	last_result = last_result->prev;
	if (last_result == NULL) {
	    /* We have freed all the results */
	    results = NULL;
	    break;
	} else {
	    last_result->next = NULL;
	}
	destroy_result(r);
    }

    /* Now find where to add the result to the time-ordered list */

    /* If the cache is empty, it becomes the first and only item */
    if (results == NULL) {
	/* Sanity check */
	if (last_result != NULL) {
	    fprintf(stderr, "results is NULL but last_result isn't\n");
	}
        result->next = NULL;
        result->prev = NULL;
	results = last_result = result;
	return result;
    }

    /* If it's after the last one, add it at the tail of the list */
    if (result->t > last_result->t) {
after_last:
	result->next = NULL;
	result->prev = last_result;
	last_result->next = result;
	last_result = result;
	return result;
    }

    /* If it's before the first one, tack it onto head of list, */
    if (result->t < results->t) {
	result->prev = NULL;
	result->next = results;
	results->prev = result;
	results = result;
	return result;
    }

    /* Scan forwards or backwards, depending whether it's closer to
     * the beginning or the end */
    if (last_result->t - result->t > result->t - results->t) {
	/* It's closer to the start than the end. Scan forwards. */
	/* Find which element to place it after and drop duplicates */
	for (r=results; r != NULL && r->t < result->t; r=r->next) {
	    /* Check for duplicates. Should never happen but does rarely. */
	    if (r->t == result->t &&
		r->fft_freq == result->fft_freq &&
		r->window == result->window) {
		/* Same params: forget the new result and return the old */
		fprintf(stderr,
			"Discarding duplicate result for %g/%g/%c\n",
			result->t / sr, result->fft_freq,
			window_key(result->window));
		destroy_result(result);
		return(r);
	    }
	}
	/* r now points to the first result whose time is >= to the new result.
	 * so insert it before that one.
	 * If r is NULL, the new result is after all the cached ones.
	 */
	if (r == NULL) {
	    result->next = NULL;
	    result->prev = last_result;
	    last_result = result;
	} else {
	    calc_t *prev_result = r->prev;

	    if (prev_result == NULL) {
		/* We're inserting it at the start of the list */
		result->prev = NULL;
		result->next = results;
		results->prev = result;
		results = result;
	    } else {
		prev_result->next = result;
		result->next = r;
		r->prev = result;
		result->prev = prev_result;
	    }
	}
    } else {
	/* It's closer to the end than the start. Scan backward from the end
	 * to find the last result that's before or at the same time as
	 * the new one. */
	for (r=last_result; r != NULL && r->t > result->t; r=r->prev) {
	    /* Check for duplicates. Should never happen but does rarely. */
	    if (r->t == result->t &&
		r->fft_freq == result->fft_freq &&
		r->window == result->window) {
		/* Same params: forget the new result and return the old */
		fprintf(stderr,
			"Discarding duplicate result for %g/%g/%c\n",
			result->t / sr, result->fft_freq,
			window_key(result->window));
		destroy_result(result);
		return(r);
	    }
	}
	/* r now points to the first result whose time is <= to the new result.
	 * so add it after that one.
	 * If r is NULL, the new result is before all the cached ones.
	 */
	if (r == NULL) {
	    result->next = results;
	    result->prev = NULL;
	    results->prev = result;
	    results = result;
	} else {
	    if (r->next == NULL) {
		/* We're adding it after the last result in the cache */
		result->next = NULL;
		result->prev = r;
		r->next = result;
		last_result = result;
	    } else {
		result->next = r->next;
		result->prev = r;
		r->next->prev = result;
		r->next = result;
	    }
	}
    }

    /* Should never happen if DELTA_{GT,LE} are mutually exclusive */
    if (r == NULL) {
	fprintf(stderr, "Oops, adding to the end of the list after a scan\n");
	goto after_last;
    }

    return result;
}

/* Return the result for the given time t, fft_freq and window function
 * or NULL if it hasn't been calculated yet, in which case we schedule it.
 * speclen==-1 or window==-1 means "I don't care for what speclen/window".
 *
 * The parameters "fftfreq" and "window" may be the current settings or
 * ANY_FFTFREQ / ANY_WINDOW, which match the result for any FFT frequency
 * or window function.
 */
calc_t *
recall_result(frames_t t, freq_t fftfreq, window_function_t window)
{
    calc_t *p;

    /* If it's earlier than the first cached result or
     * later than the last cached result, we don't have it.
     * This saves uselessly scanning the whole list of results.
     */
    if (results == NULL || t < results->t)
	return NULL;
    if (last_result == NULL || t > last_result->t)
	return NULL;

    /* If it's closer in time to the first result in the cache, scan forwards
     * from the start of the cache. If it's closer in time to the last result
     * in the cache, scan backwards from the end. This reduces % of total CPU
     * spent in recall_result() from 13.5% to 1.2%.
     */
    if (t - results->t < last_result->t - t) {
	for (p=results; p != NULL; p=p->next) {
	    /* If the time is the same and fft_freq and window are the
	     * same (or ANY) then this is the result we want */
	    if (p->t == t &&
		(fftfreq == ANY_FFTFREQ || p->fft_freq == fftfreq) &&
		(window  == ANY_WINDOW  || p->window  == window)) {
		break;
	    }
	    /* If we've scanned past the requested time, it isn't there. */
	    if (p->t > t) return NULL;
	}
    } else { /* Scan backwards from the end */
	for (p=last_result; p != NULL; p=p->prev) {
	    /* If the time is the same and fft_freq and window are the
	     * same (or ANY) then this is the result we want */
	    if (p->t == t &&
		(fftfreq == ANY_FFTFREQ || p->fft_freq == fftfreq) &&
		(window  == ANY_WINDOW  || p->window  == window)) {
		break;
	    }
	    /* If we've scanned past the requested time, it isn't there. */
	    if (p->t < t) return NULL;
	}
    }

    return p;	/* NULL if not found */
}

/* Forget the result cache */
void
drop_all_results(void)
{
    calc_t *r;

    for (r = results; r != NULL; /* see below */) {
	calc_t *next = r->next;
	destroy_result(r);
	r = next;
    }
    results = last_result = NULL;
}

/* Free the memory associated with a result structure */
static void
destroy_result(calc_t *r)
{
    free(r->spec);
    free(r);
}

/* Show stats about the result cache, for debugging:
 * For each different FFT-size/window, show the range of results that are
 * cached. We do this by having our own list of calc_t's, one for each
 * different combination of FFT-size and window, and keeping a singly
 * linked list of 
 */
typedef struct result_type {
    freq_t fft_freq;
    window_function_t window;
    unsigned n_results;
    frames_t earliest;
    frames_t latest;
    struct result_type *next;
} result_type_t;

void
print_result_cache_stats(void)
{
    result_type_t *types = NULL;
    calc_t *r;
    result_type_t *t;

    for (r=results; r != NULL; r=r->next) {
	/* Update the appropriate result type */
	for (t=types; t != NULL; t=t->next) {
	    if (t->fft_freq == r->fft_freq && t->window == r->window) {
		/* Found the right type. Update it. */
		t->n_results++;
		if (r->t < t->earliest) t->earliest = r->t;
		if (r->t > t->latest)   t->latest = r->t;
		break;
	    }
	}
	if (t == NULL) {
	    /* We didn't find a result_type of this kind so make a new one */
	    result_type_t *new_type = Malloc(sizeof(result_type_t));
	    new_type->fft_freq = r->fft_freq;
	    new_type->window = r->window;
	    new_type->earliest = new_type->latest = r->t;
	    new_type->n_results = 1;
	    /* Tack onto head of list bcos it's easiest */
	    new_type->next = types;
	    types = new_type;
	}
    }

    /* Now, for each result type, print its stats */
    printf("fft_freq window  num  from  to\n");
    for (t=types; t != NULL; t=t->next) {
	printf("%8g %6c %5d %5s",
	       t->fft_freq, window_key(t->window), t->n_results,
	       frames_to_string(t->earliest));
	printf(" %5s\n",
	       frames_to_string(t->latest));
    }

    for (t=types; t != NULL; /* below */) {
	result_type_t *next_t = t->next;
	free(t);
	t = next_t;
    }
}
