/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * timer.c - The timer is used to keep scrolling the display
 */

#include "spettro.h"
#include "timer.h"
#include "gui.h"
#include "paint.h"		/* for do_scroll() */

/* The timer and its callback function. */
#if ECORE_TIMER

#include <Ecore.h>
#include <Evas.h>
typedef Ecore_Timer * timer_type;
#define NO_TIMER NULL

extern Evas_Object *em;	/* From main.c */

static Eina_Bool timer_cb(void *data);	/* The timer callback function */
static int scroll_event;   /* Our user-defined event to activate scrolling */
static Eina_Bool scroll_cb(void *data, int type, void *event);

#elif SDL_TIMER

# include <SDL.h>

typedef Uint32 timer_type;
#define NO_TIMER 0
static Uint32 timer_cb(Uint32 interval, void *data);

#else
# error "Define ECORE_TIMER or SDL_TIMER"
#endif

static timer_type timer = NO_TIMER;

/* Implementation-specific code used by the public functions */

static void
add_timer(secs_t interval)
{
    /* We limit the scrolling rate to some minimum to avoid GUI death
     * at microscopic intervals.
     * If we can find out the monitor's refresh rate, use that instead */
    double minimum_interval = 1/50.0;	/* typical CRT frame rate */

    if (interval < minimum_interval) interval = minimum_interval;

#if ECORE_TIMER
    /* The timer callback just generates an event, which is processed in
     * the main ecore event loop to do the scrolling in the main loop
     */
    scroll_event = ecore_event_type_new();
    ecore_event_handler_add(scroll_event, scroll_cb, NULL);
    timer = ecore_timer_add((double)interval, timer_cb, (void *)em);
#elif SDL_TIMER
    timer = SDL_AddTimer((Uint32)round(interval * 1000), timer_cb, (void *)NULL);
#endif
    if (timer == NO_TIMER) {
	fprintf(stderr, "Couldn't add a timer for an interval of %g secs.\n", interval);
	exit(1);
    }
}

static void
delete_timer()
{
#if ECORE_MAIN
    (void) ecore_timer_del(timer);
#elif SDL_MAIN
    SDL_RemoveTimer(timer);
#endif
}

/* Public functions */

extern freq_t fps;	/* From main.c */

void
start_timer()
{
    /* Start screen-updating and scrolling timer */
    add_timer(1.0/fps);
}

void
stop_timer()
{
    delete_timer();
}

void
change_timer_interval(secs_t interval)
{
    delete_timer();
    add_timer((double)interval);
}

/*
 * The periodic timer callback that, when playing, schedules scrolling of
 * the display by one pixel.
 * When paused, the timer continues to run to update the display in response to
 * seek commands.
 */

/* Implementation-specific code to handle the timer callback */

#if ECORE_TIMER
    static Eina_Bool
    timer_cb(void *data)
#elif SDL_TIMER
    static Uint32
    timer_cb(Uint32 interval, void *data)
#endif
{

#if ECORE_TIMER
    /* Generate a user-defined event which will be processed in the main loop */
    ecore_event_add(scroll_event, NULL, NULL, NULL);

    return ECORE_CALLBACK_RENEW;

#elif SDL_TIMER

    /* We only want one scroll event pending at a time, otherwise if there's
     * insufficient CPU, the event queue fills up with them and other events
     * stop working too (result events, key presses etc)
     */
    {
	SDL_Event event;

	event.type = SDL_USEREVENT;
	event.user.code = SCROLL_EVENT;
	if (SDL_PushEvent(&event) != SDL_PUSHEVENT_SUCCESS)
	    fprintf(stderr, "Couldn't push an SDL scroll event\n");
    }

    return(interval);

#endif
}

#if ECORE_TIMER

static Eina_Bool
scroll_cb(void *data, int type, void *event)
{
    do_scroll();
    return ECORE_CALLBACK_DONE;
}

#endif
