/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2025 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#ifndef CONTOURWIDGET_H
#define CONTOURWIDGET_H

#include <Action.h>
#include <MeshComponent.h>
#include <ImageComponent.h>
#include <Transformation.h>

// disable warning generated by clang about the surrounded headers
#include <CamiTKDisableWarnings>
#include <vtkSmartPointer.h>
#include <vtkContourWidget.h>
#include <vtkBoundedPlanePointPlacer.h>
#include <CamiTKReEnableWarnings>

#include <QObject>
#include <QFrame>
#include <QMap>

/** Adding a contour widget to the currently selected image component
 * To simplify this class inherits from both CamiTK Action and VTK Command, for real life code, please
 * add a specific/separated class inheriting from vtkCommand.
 */
class ContourWidget : public camitk::Action {
    Q_OBJECT

public:
    /// Default Constructor
    ContourWidget(camitk::ActionExtension* extension);

    /// Default Destructor
    virtual ~ContourWidget() = default;

    /// method called when the action when the action is triggered (i.e. started)
    virtual QWidget* getWidget();

public slots:
    /** this method is automatically called when the action is triggered.
      * Call getTargets() method to get the list of components to use.
      * \note getTargets() is automatically filtered so that it only contains compatible components,
      * i.e., instances of ImageComponent (or a subclass).
      */
    virtual camitk::Action::ApplyStatus apply();

    /// close the current contour
    void closeContour();

    /// delete current contour, connect to current image and current mesh destroyed() signal
    void deleteContour();

    /// update the action GUI (= action widget) information (number of points)
    void update();

    // duplicate the current contour but start a new mesh
    void duplicateContour();

private:
    /// this action widget
    QFrame* myWidget;

    /// currently selected image component
    camitk::ImageComponent* currentImage;

    /// the mesh component build by the contour widget.
    /// Its frame is the same as the data image frame.
    camitk::MeshComponent* currentMesh;

    /// the interactive contour representation
    vtkSmartPointer<vtkContourWidget> contourWidget;

    /// Control the user interaction with the contour widget
    /// Produces the vtkPointSet that will be both:
    /// - the input of the transformFilter to display the contour in the 2D viewers
    /// - the MeshComponent vtkPointSet that will itself create its actors for the 3D viewer
    vtkSmartPointer<vtkCommand> contourWidgetCommand;

    /// placer for the contour representation that must
    /// - be fixed to the normal plane of the selected 2D viewer
    /// - move with the selected slice id
    /// the translation from the data frame of the image to the current slice
    /// is expressed in the data frame (e.g for the first slice the translation in 0,
    /// for the last slice it is equal to the nbVoxel*voxelSize (in the normal plane of
    /// the current 2D viewer)
    /// This placer is needed in order to show the vtkContourWidget at the correct
    /// place inside the 2D viewer and will be used by the contour representation
    /// to create the mesh at the right place in space.
    vtkSmartPointer<vtkBoundedPlanePointPlacer> frontPlanePlacer;

    /// The 2D viewer currently selected (0=axial, 1=coronal and 2=sagittal)
    int selectedViewer;

    /// initialize the contour widget
    /// @return false if the contour could not be created, true otherwise
    bool initContour();

    /// create the MeshComponent from the current contour
    void initMesh();

    /// update the contour visualization (vtkContourWidget + MeshComponent)
    void updateContourAndMesh();

    /// compute the current transformation (translation from the image main frame to the current slice)
    void updateTransformation();

    /// get the viewer string corresponding to the current selected viewer name
    /// axial = 0, coronal = 1, sagittal = 2
    QString getViewerName();

    /// get the viewer string list
    QStringList getViewerNameList();

};

#endif // CONTOURWIDGET_H
