/**
 * Copyright (C) 2009-2013 Paul Fretwell - aka 'Footleg' (drfootleg@gmail.com)
 * 
 * This file is part of Cave Converter.
 * 
 * Cave Converter is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Cave Converter is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Cave Converter.  If not, see <http://www.gnu.org/licenses/>.
 */
package footleg.cavesurvey.cmdline;

import java.io.File;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

import footleg.cavesurvey.data.model.CaveSurvey;
import footleg.cavesurvey.data.reader.DxfParser;
import footleg.cavesurvey.data.reader.PocketTopoParser;
import footleg.cavesurvey.data.reader.SurvexParser;
import footleg.cavesurvey.data.writer.CompassWriter;
import footleg.cavesurvey.data.writer.SurvexWriter;
import footleg.cavesurvey.data.writer.TopoRobotWriter;
import footleg.cavesurvey.tools.UtilityFunctions;

/**
 * Command line application for converting cave survey data from one file format to another.
 * Currently it can read some survex files, text export files from PocketTopo and DXF format
 * data. It can write toporobot and survex files.
 *  
 * @author      Footleg <drfootleg@gmail.com>
 * @version     2013.07.18                                (ISO 8601 YYYY.MM.DD)
 * @since       1.6                                       (The Java version used)
 */
public class CaveConverter {
	private static List<String> log = new ArrayList<String>();
	private static final String filePath = "./";
	private static Date today;
	
	public static String newline = System.getProperty("line.separator");

	/**
	 * Command line options for True, False or 'Use Default'
	 */
	public enum CmdlineOpt {
	    T, F, D 
	}

    /**
	 * @param args Command line arguments
	 */
	public static void main( String[] args ) {
		//Default input and output filenames
		String inputFilename = "input.svx";
		String outputFilename = "output.text";
		char inputFormat = 'x'; //x = not valid
		char outputFormat = 'x'; //x = not valid
		
		//Set date if not already set
		if ( today == null ) {
			//Set date to todays datetime
			Calendar cal = Calendar.getInstance(TimeZone.getDefault());
			today = cal.getTime();
		}
		//Check for arguments
		if (args.length > 0) {
			//Set input filename from 1st argument
			inputFilename = args[0];
			if (args.length > 1) {
				//Set output filename from 2nd argument
				outputFilename = args[1];
				if (args.length > 2) {
					//Set input format from 3rd argument
					try {
						inputFormat = parseSingleCharacterArgument( args[2] );
						if (args.length > 3) {
							//Set input format from 4th argument
							outputFormat = parseSingleCharacterArgument( args[3] );
							
							//Set options flags from 5th argument onwards
							CmdlineOpt splaysOpt = CmdlineOpt.D;
							CmdlineOpt genLRUDOpt = CmdlineOpt.F;
							for (int iOpts = 4; iOpts < args.length; iOpts++ ) {
								if ( args[iOpts].compareToIgnoreCase("nosplays") == 0 ) {
									splaysOpt = CmdlineOpt.F;
								}
								else if ( args[iOpts].compareToIgnoreCase("splays") == 0 ) {
									splaysOpt = CmdlineOpt.T;
								}
								if ( args[iOpts].compareToIgnoreCase("lrud") == 0 ) {
									genLRUDOpt = CmdlineOpt.T;
								}
							}

							//Call file convert method
							convertFile(inputFilename, outputFilename, inputFormat, outputFormat, splaysOpt, genLRUDOpt );
						}
					} catch (ParseException e) {
						//Log error
						e.printStackTrace();
					}
				}
			}
		}
	}
	
	public static void setToday(Date date) {
		CaveConverter.today = date;
	}

	/**
	 * Parses a string argument into a single character.
	 *
	 * @param  argument String to be converting into a single character
	 * @throws ParseException If the argument is not a single string
	 */
	private static char parseSingleCharacterArgument( String argument ) throws ParseException {
		char[] chars = argument.toCharArray();
		char argChar = 'x';
		
		if ( chars.length == 1 ) {
			argChar = chars[0];
		}
		else {
			ParseException e = new ParseException("Failed to parse argument to single character. " +
				"Argument was '" + argument + "'.", 1);
			throw e;
		}
		
		return argChar;
	}
	
	/**
	 * Reads in a survey data file and converts it to another format which is written out to a file.
	 *
	 * @param  inputFilename   Name and location of file to be converted
	 * @param  outputFilename  Name and location of file to be output
	 * @param  inputFormat     Format of input file
	 * @param  outputFormat    Format of file to be generated
	 * @param  splaysOpt       Indicates if a splays output option was set to True, False or to use default for writer (T,F,D)
	 * @param  generateLRUDOpt Indicates whether to generate LRUD data from splays before writing out data
	 * @throws ParseException 
	 */
	public static void convertFile( String inputFilename, String outputFilename, char inputFormat, 
			char outputFormat, CmdlineOpt splaysOpt, CmdlineOpt generateLRUDOpt ) throws ParseException {
		//Declare structure to hold survey data
		CaveSurvey surveyData = null;
		
		//Read input data file
		CaveConverter.logMessage( "Reading data file: " + inputFilename );
		List<String> fileData = UtilityFunctions.readTextFile( new File( inputFilename ) );
		
		//Parse file data
		if ( inputFormat == UtilityFunctions.survexFormat ) {
			//Parse Survex data
			SurvexParser parser = new SurvexParser();
			surveyData = parser.parseSurvexFile( fileData );
		}
		else if ( inputFormat == UtilityFunctions.pocketTopoFormat ) {
			//Parse PocketTopo data
			PocketTopoParser parser = new PocketTopoParser();
			surveyData = parser.parseFile( fileData );
		}
		else if ( inputFormat == UtilityFunctions.dxfFormat ) {
			//Parse Autocad DXF data polylines into survey series
			DxfParser parser = new DxfParser();
			surveyData = parser.parseFile(fileData, 0);
		}
		else {
			//Unsupported input format argument
			logMessage("Unsupported input format argument: " + inputFormat + " is not a valid input format.");
		}
		
		//Set options flag for generating LRUD data
		boolean generateLRUD = false;
		if (generateLRUDOpt == CmdlineOpt.T) {
			generateLRUD = true;
		}
		
		//Generate LRUD data if required
		if ( generateLRUD ) {
			surveyData.generateLRUDfromSplays();
		}
		
		//Convert data to output format
		List<String> outputData = null;
		if ( surveyData != null ) {
			if ( outputFormat == UtilityFunctions.compassFormat ) {
				//Generate Compass format data
				CompassWriter writer = new CompassWriter();
				outputData = writer.generateCompassData( surveyData );
			}
			else if ( outputFormat == UtilityFunctions.survexFormat ) {
				//Set options flag for splays (default to true)
				boolean outputSplays = true;
				if (splaysOpt == CmdlineOpt.F) {
					outputSplays = false;
				}
				//Generate Survex format data
				SurvexWriter writer = new SurvexWriter();
				outputData = writer.generateSurvexData( surveyData, outputSplays );
			}
			else if ( outputFormat == UtilityFunctions.toporobotFormat ) {
				//Set options flag for splays (default to false)
				boolean outputSplays = false;
				if (splaysOpt == CmdlineOpt.T) {
					outputSplays = true;
				}
				//Generate Toporobot format data
				TopoRobotWriter writer = new TopoRobotWriter();
				outputData = writer.generateToporobotData( surveyData, today, outputSplays );
			}
			else {
				//Unsupported output format argument
				logMessage("Unsupported output format argument: " + outputFormat + " is not a valid output format.");
			}
		}
		
		//Write output file
		if ( outputData.size() > 0 ) {
			String outputFilePath = filePath + outputFilename;
			CaveConverter.logMessage( "Writing output file: " + outputFilePath );
			String error = UtilityFunctions.writeTextFile(outputData, outputFilePath);
			if ( error.length() > 0 ) {
				logMessage( error );
			}
		}
		
		//Write log file
		UtilityFunctions.writeTextFile(log, filePath + "CaveConverter.log");
	}
	
	/**
	 * logMessage
	 * 
	 * Formats message with date time stamp, then outputs to console and log
	 */
	public static void logMessage( String message ) {
		String logLine;

		Calendar cal = Calendar.getInstance(TimeZone.getDefault());
		String DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";
		SimpleDateFormat sdf = new SimpleDateFormat(DATE_FORMAT);	
		sdf.setTimeZone(TimeZone.getDefault());          

		logLine = sdf.format(cal.getTime()) + ": " + message;
		log.add( logLine );
		System.out.println(logLine);

	}
	
	public static String padNumber(int num, int padWidth) {
		String numStr = "" + num;
		
		return padString(numStr,padWidth);
	}
	
	public static String padNumber(double num, int decPlaces, int padWidth) {
		String formatStr = "0.";
		for (int i = 0; i < decPlaces; i++) {
			formatStr += '0';
		}
		DecimalFormat formatter = new DecimalFormat(formatStr);
		
		return padString(formatter.format(num), padWidth);
	}
	
	public static String padString(String data, int padWidth) {
		String padded;
		//Pad number with spaces if not already as wide as pad width
		if ( data.length() < padWidth ) {
			//Pad with lots of spaces
			padded = "                    " + data;
			//then trim to padded width
			padded = padded.substring(padded.length() - padWidth);
		}
		else {
			//Already wide enough, so just return as is
			padded = data;
		}
		
		return padded;
	}
	
}
