#
# Copyright CEA/DAM/DIF (2007-2015)
#  Contributor: Stephane THIELL <sthiell@stanford.edu>
#
# This file is part of the ClusterShell library.
#
# This software is governed by the CeCILL-C license under French law and
# abiding by the rules of distribution of free software.  You can  use,
# modify and/ or redistribute the software under the terms of the CeCILL-C
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
#
# As a counterpart to the access to the source code and  rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty  and the software's author,  the holder of the
# economic rights,  and the successive licensors  have only  limited
# liability.
#
# In this respect, the user's attention is drawn to the risks associated
# with loading,  using,  modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean  that it is complicated to manipulate,  and  that  also
# therefore means  that it is reserved for developers  and  experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and,  more generally, to use and operate it in the
# same conditions as regards security.
#
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL-C license and that you accept its terms.

"""
ClusterShell Event handling.

This module contains the base class **EventHandler** which defines a simple
interface through methods to handle events coming from ClusterShell I/O Engine
clients. Events are generated by Worker, EngineTimer or EnginePort objects.
"""

class EventHandler(object):
    """ClusterShell EventHandler interface.

    Derived class should implement the following methods to listen for Worker,
    EngineTimer or EnginePort chosen events.
    """

    def ev_start(self, worker):
        """
        Called to indicate that a worker has just started.

        :param worker: :class:`.Worker` object
        """

    def ev_pickup(self, worker):
        """
        Called to indicate that a worker command for a specific node (or key)
        has just started. Called for each node.

        :param worker: :class:`.Worker` object

        Available worker attributes:

        * :attr:`.Worker.current_node` - node (or key)
        """

    def ev_read(self, worker):
        """
        Called to indicate that a worker has data to read from a specific
        node (or key).

        :param worker: :class:`.Worker` object

        Available worker attributes:

        * :attr:`.Worker.current_node` - node (or key)
        * :attr:`.Worker.current_msg` - read message
        """

    def ev_error(self, worker):
        """
        Called to indicate that a worker has error to read on stderr from
        a specific node (or key).

        :param worker: :class:`.Worker` object

        Available worker attributes:

        * :attr:`.Worker.current_node` - node (or key)
        * :attr:`.Worker.current_errmsg` - read error message
        """

    def ev_written(self, worker, node, sname, size):
        """
        Called to indicate that some writing has been done by the worker to a
        node on a given stream. This event is only generated when ``write()``
        is previously called on the worker.

        This handler may be called very often depending on the number of target
        nodes, the amount of data to write and the block size used by the
        worker.

        Note: up to ClusterShell 1.6, this event handler wasn't implemented. To
        properly handle ev_written after 1.6, the method signature must consist
        of the following parameters:

        :param worker: :class:`.Worker` object
        :param node: node (or) key
        :param sname: stream name
        :param size: amount of bytes that has just been written to node/stream
            associated with this event
        """

    def ev_hup(self, worker):
        """
        Called to indicate that a worker command for a specific node (or key)
        has just finished. Called for each node.

        :param worker: :class:`.Worker` object

        Available worker attributes:

        * :attr:`.Worker.current_node` - node (or key)
        * :attr:`.Worker.current_rc` - command return code
        """

    def ev_timeout(self, worker):
        """
        Called to indicate that a worker has timed out (worker timeout only).

        :param worker: :class:`.Worker` object
        """

    def ev_close(self, worker):
        """
        Called to indicate that a worker has just finished (it may already
        have failed on timeout).

        :param worker: :class:`.Worker` object
        """

    def ev_msg(self, port, msg):
        """
        Called to indicate that a message has been received on an EnginePort.

        Used to deliver messages reliably between tasks.

        :param port: EnginePort object on which a message has been received
        :param msg: the message object received
        """

    def ev_timer(self, timer):
        """
        Called to indicate that a timer is firing.

        :param timer: :class:`.EngineTimer` object that is firing
        """

    def _ev_routing(self, worker, arg):
        """
        Routing event (private). Called to indicate that a (meta)worker has just
        updated one of its route path. You can safely ignore this event.
        """
