/*
 * libcolorblind - Pixel Filter for colorblind accessibility
 *
 * This is in Public Domain
 * Original Author: Daniel Ruoso <daniel@ruoso.com>
 *
 * This library provides an unified way to recalculate colors
 * in order to present alternative views on images for colorblind
 * people.
 *
 */
#include <stdio.h>
#include "colorblind.h"
#include "filters.h"
#include "utils.h"

#define level_sensibility 40

void _colorblind_calc_all_values(COLORBLIND_XCOLOR* color, int* max, int* mid, int* min,
                                 int* max_count, int* min_count) {
        *max = 0;
        *mid = -1;
        *min = 255;
        *max_count = 0;
        *min_count = 0;

        // get max and min
        if (color->red > *max) *max = color->red;
        if (color->green > *max) *max = color->green;
        if (color->blue > *max) *max = color->blue;
        if (color->red < *min) *min = color->red;
        if (color->green < *min) *min = color->green;
        if (color->blue < *min) *min = color->blue;

        // count max and min
        if (color->red == *min) {
                (*min_count)++;
        }
        if (color->red == *max) {
                (*max_count)++;
        }
        if (color->green == *min) {
                (*min_count)++;
        }
        if (color->green == *max) {
                (*max_count)++;
        }
        if (color->blue == *min) {
                (*min_count)++;
        }
        if (color->blue == *max) {
                (*max_count)++;
        }
        if (*max_count == 1 && *min_count == 1) {
                // find the mid
                if (color->red > *min && color->red < *max) *mid = color->red;
                if (color->green > *min && color->green < *max) *mid = color->green;
                if (color->blue > *min && color->blue < *max) *mid = color->blue;
        } else if (*max_count > 1) {
                *mid = *max;
        } else if (*min_count > 1) {
                *mid = *min;
        }
}

int colorblind_get_num_equalized(COLORBLIND_RUNTIME* cbr, COLORBLIND_XCOLOR* color) {
        int max,mid,min,max_count,min_count;
        _colorblind_calc_all_values(color,&max,&mid,&min,&max_count,&min_count);

        if (max_count == 2 || min_count == 2) {
                // two are equals
                return 2;
        } else if (max_count > 2 || min_count > 2) {
                // three are equals
                return 3;
        }

        if (max - min <= level_sensibility) {
                // all colors equalized.
                return 3;
        }
        
        if (max - mid <= level_sensibility || mid - min <= level_sensibility) {
                // two colors equalized.
                return 2;
        }
        
        return 1;
}

int colorblind_get_num_main(COLORBLIND_RUNTIME* cbr, COLORBLIND_XCOLOR* color) {
        int max,mid,min,max_count,min_count;
        _colorblind_calc_all_values(color,&max,&mid,&min,&max_count,&min_count);
        if (max_count == 3) {
                return 3;
        } else if (max - min <= level_sensibility) {
                return 3;
        } else if (max - mid <= level_sensibility) {
                return 2;
        } else {
                return 1;
        }
}

int _colorblind_x_is_main(COLORBLIND_RUNTIME* cbr, COLORBLIND_XCOLOR* color, int x) {
        int max,mid,min,max_count,min_count;
        _colorblind_calc_all_values(color,&max,&mid,&min,&max_count,&min_count);
        if (max_count == 3) {
                return 1;
        } else if (max - min <= level_sensibility) {
                return 1;
        } else if (max - mid <= level_sensibility) {
                if (x == max || x == mid) {
                        return 1;
                } else {
                        return 0;
                }
        } else {
                if (x == max) {
                        return 1;
                } else {
                        return 0;
                }
        }
}

int colorblind_red_is_main(COLORBLIND_RUNTIME* cbr, COLORBLIND_XCOLOR* color) {
        return _colorblind_x_is_main(cbr,color,color->red);
}

int colorblind_green_is_main(COLORBLIND_RUNTIME* cbr, COLORBLIND_XCOLOR* color) {
        return _colorblind_x_is_main(cbr,color,color->green);
}

int colorblind_blue_is_main(COLORBLIND_RUNTIME* cbr, COLORBLIND_XCOLOR* color) {
        return _colorblind_x_is_main(cbr,color,color->blue);
}

