/**
* @file  WfLogDispatcher.cc
* 
* @brief Workflow Log Dispatcher
* 
* @author  Benjamin Isnard (benjamin.isnard@ens-lyon.fr)
* 
* @section Licence
*   |LICENCE|                                                                
*/

#include "WfLogDispatcher.hh"

using events::EventFrom;
using events::EventStandardMsg;
using events::EventCreateObject;
using events::EventCreateDirectedLink;

WfLogDispatcher::WfLogDispatcher(std::ostream& output)
  : myOutput(output) {
  myWfLogSrv = WfLogService::_nil();
  registerEventFunc(this, &WfLogDispatcher::onWorkflowInstanciate);
  registerEventFunc(this, &WfLogDispatcher::onWorkflowInstDone);
  registerEventFunc(this, &WfLogDispatcher::onWorkflowInstError);
  registerEventFunc(this, &WfLogDispatcher::onWorkflowCompleted);
  registerEventFunc(this, &WfLogDispatcher::onWorkflowCancelled);
  registerEventFunc(this, &WfLogDispatcher::onEventCreateWorkflow);
  registerEventFunc(this, &WfLogDispatcher::onEventCreateWfDag);
  registerEventFunc(this, &WfLogDispatcher::onEventCreateDagNode);
  registerEventFunc(this, &WfLogDispatcher::onEventCreateDagNodePort);
  //   registerEventFunc(this, &WfLogDispatcher::onEventCreateDagNodeLink);
  registerEventFunc(this, &WfLogDispatcher::onInPortDepend);
  registerEventFunc(this, &WfLogDispatcher::onDataElements);
  registerEventFunc(this, &WfLogDispatcher::onDagNodeSetDataID);
  registerEventFunc(this, &WfLogDispatcher::onDagChangeId);
  registerEventFunc(this, &WfLogDispatcher::onDagStateChange);
  registerEventFunc(this, &WfLogDispatcher::onDagNodeReady);
  registerEventFunc(this, &WfLogDispatcher::onDagNodeStart);
  registerEventFunc(this, &WfLogDispatcher::onDagNodeFinish);
  registerEventFunc(this, &WfLogDispatcher::onDagNodeFailed);
  registerEventFunc(this, &WfLogDispatcher::onSourceDataTree);
  registerEventFunc(this, &WfLogDispatcher::onSinkData);
}

bool
WfLogDispatcher::isOnline() {
  return !CORBA::is_nil(myWfLogSrv);
}

void
WfLogDispatcher::setWfLogService(WfLogService_var logService) {
  myWfLogSrv = logService;
  myOutput << "WFLOGSERVICE DEFINED\n";
}

/** WORKFLOWS **/
void
WfLogDispatcher::onEventCreateWorkflow(
  const EventCreateObject< FWorkflow, FWorkflow >* event) {
  myOutput << "WF CREATE: " << *event << "\n";
  if (isOnline()) {
    std::string rootWfId = (event->getParent() == NULL) ?
      "" : event->getParent()->getRootWorkflow()->getId();
    myWfLogSrv->initWorkflow(event->getObject()->getId().c_str(),
                             event->getObject()->getName().c_str(),
                             rootWfId.c_str());
  }
}

// following workflow events are sent only for the root workflow
// (events generated by CltWfMgr) and not for sub-workflows
void
WfLogDispatcher::onWorkflowInstanciate(
  const EventFrom<FWorkflow,
  EventStandardMsg<FWorkflow,
  FWorkflow::INST > >* event) {
  myOutput << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->updateWorkflow(event->getSource()->getId().c_str(),
                               "INST", "");
  }
}

void
WfLogDispatcher::onWorkflowInstDone(
  const EventFrom<FWorkflow,
  EventStandardMsg<FWorkflow,
  FWorkflow::INSTDONE > >*
  event) {
  myOutput << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->updateWorkflow(event->getSource()->getId().c_str(),
                               "INSTDONE", "");
  }
}

void
WfLogDispatcher::onWorkflowInstError(
  const EventFrom<FWorkflow,
  EventStandardMsg<FWorkflow,
  FWorkflow::INSTERROR > >*
  event) {
  myOutput << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->updateWorkflow(event->getSource()->getId().c_str(),
                               "INSTERROR", "");
  }
}

void
WfLogDispatcher::onWorkflowCancelled(
  const EventFrom<FWorkflow,
  EventStandardMsg<FWorkflow, FWorkflow::CANCELLED > >* event) {
  myOutput << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->updateWorkflow(event->getSource()->getId().c_str(),
                               "CANCELLED", "");
  }
}

void
WfLogDispatcher::onWorkflowCompleted(
  const EventFrom<FWorkflow,
  EventStandardMsg<FWorkflow, FWorkflow::COMPLETED > >* event) {
  myOutput << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->updateWorkflow(event->getSource()->getId().c_str(),
                               "COMPLETED", "");
  }
}

void
WfLogDispatcher::onEventCreateWfDag(
  const EventCreateObject<Dag, FWorkflow>* event) {
  myOutput << "DAG CREATE: " << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->createDag(event->getObject()->getId().c_str(),
                          event->getParent()->getId().c_str());
  }
}

void
WfLogDispatcher::onSourceDataTree(
  const EventFrom<FSourceNode,
  EventStandardMsg<FSourceNode,
  FSourceNode::DATATREE > >* event) {
  myOutput << "SOURCE DATA TREE: " << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->createSourceDataTree(event->getSource()->getId().c_str(),
                                     event->getSource()->getWorkflow()->
                                     getId().c_str(),
                                     event->getData().c_str());
  }
}

void
WfLogDispatcher::onSinkData(
  const EventFrom<FSinkNode,
  EventStandardMsg<FSinkNode, FSinkNode::DATAID > >* event) {
  myOutput << "SINK DATA : " << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->createSinkData(event->getSource()->getId().c_str(),
                               event->getSource()->getWorkflow()->
                               getId().c_str(),
                               event->getData().c_str());
  }
}

/** DAGS **/

void
WfLogDispatcher::onDagChangeId(
  const EventFrom<Dag, EventStandardMsg<Dag, Dag::MODID > >* event) {
  myOutput << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->updateDag(event->getSource()->getId().c_str(),
                          event->getSource()->getWorkflow()->getId().c_str(),
                          "IDCHANGE",
                          event->getData().c_str());
  }
}

void
WfLogDispatcher::onDagIsEmpty(
  const EventFrom<Dag, EventStandardMsg<Dag, Dag::EMPTY > >* event) {
  myOutput << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->updateDag(event->getSource()->getId().c_str(),
                          event->getSource()->getWorkflow()->getId().c_str(),
                          "EMPTYDAG",
                          event->getData().c_str());
  }
}

void
WfLogDispatcher::onDagStateChange(
  const EventFrom<Dag, EventStandardMsg<Dag, Dag::STATE > >* event) {
  myOutput << *event << "\n";
  // currently only dag cancelled event is generated
}

void
WfLogDispatcher::onEventCreateDagNode(
  const EventCreateObject<DagNode, Dag>* event) {
  myOutput << "DAGNODE CREATE: " << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->createDagNode(event->getObject()->getId().c_str(),
                              event->getObject()->getDag()->getId().c_str(),
                              event->getObject()->getDag()->
                              getWorkflow()->getId().c_str());
  }
}

/** DAG NODES **/

void
WfLogDispatcher::onEventCreateDagNodePort(
  const EventCreateObject<DagNodePort, DagNode>* event) {
  myOutput << "DN PORT CREATE: " << *event << "\n";
  if (isOnline()) {
    std::string portDir;
    switch (event->getObject()->getPortType()) {
    case WfPort::PORT_IN:
      portDir = "IN";
      break;
    case WfPort::PORT_OUT:
      portDir = "OUT";
      break;
    case WfPort::PORT_INOUT:
      portDir = "INOUT";
      break;
    default:
      portDir = "";
    }

    myWfLogSrv->createDagNodePort(event->getObject()->getId().c_str(),
                                  portDir.c_str(),
                                  event->getParent()->getId().c_str(),
                                  event->getParent()->getDag()->
                                  getWorkflow()->getId().c_str());
  }
}

void
WfLogDispatcher::onInPortDepend(
  const EventFrom<DagNodeInPort,
  EventStandardMsg<DagNodeInPort,
  DagNodeInPort::DEPEND > >* event) {
  myOutput << "INPORT DEPEND: " << *event << "\n";
  if (isOnline()) {
    DagNode* srcNode = dynamic_cast<DagNode*>(event->getSource()->getParent());
    myWfLogSrv->
      setInPortDependencies(event->getSource()->getId().c_str(),
                            srcNode->getId().c_str(),
                            srcNode->getDag()->getWorkflow()->getId().c_str(),
                            event->getData().c_str());
  }
}

void
WfLogDispatcher::onDataElements(
  const EventFrom<WfDataIDAdapter,
  EventStandardMsg<WfDataIDAdapter,
  WfDataIDAdapter::ELTIDLIST > >* event) {
  myOutput << "DATA ELEMENTS: " << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->createDataElements(event->getSource()->getDataID().c_str(),
                                   event->getData().c_str());
  }
}

void
WfLogDispatcher::onDagNodeSetDataID(
  const EventFrom<DagNodePort,
  EventStandardMsg<DagNodePort,
  DagNodePort::DATAID > >* event) {
  myOutput << "SET DATAID: " << *event << "\n";
  if (isOnline()) {
    const DagNodePort *srcPort = event->getSource();
    const DagNode *srcNode = dynamic_cast<const DagNode*>(srcPort->getParent());
    myWfLogSrv->
      createDagNodeData(srcNode->getId().c_str(),
                        srcNode->getDag()->getWorkflow()->getId().c_str(),
                        srcPort->getId().c_str(), event->getData().c_str());
  }
}


void
WfLogDispatcher::onEventCreateDagNodeLink(
  const EventCreateDirectedLink<DagNode, DagNode>* event) {
  myOutput << "DAGNODE LINK: " << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->createDagNodeLink(event->getSource()->getId().c_str(),
                                  event->getSource()->getDag()->
                                  getWorkflow()->getId().c_str(),
                                  event->getDestination()->getId().c_str(),
                                  event->getDestination()->getWorkflow()
                                  ->getId().c_str());
  }
}

void
WfLogDispatcher::onDagNodeReady(
  const EventFrom<DagNode,
  EventStandardMsg<DagNode,
  DagNode::READY > >* event) {
  myOutput << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->nodeIsReady(event->getSource()->getId().c_str(),
                            event->getSource()->getDag()->getWorkflow()->
                            getId().c_str());
  }
}

void
WfLogDispatcher::onDagNodeStart(
  const EventFrom<DagNodeLauncher,
  EventStandardMsg<DagNodeLauncher,
  DagNode::START > >* event) {
  myOutput << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->nodeIsStarting(event->getSource()->getNode()->getId().c_str(),
                               event->getSource()->getNode()->getDag()->
                               getWorkflow()->getId().c_str(),
                               event->getSource()->getNode()->
                               getPbName().c_str(),
                               event->getSource()->getSeDName().c_str());
  }
}

void
WfLogDispatcher::onDagNodeFinish(
  const EventFrom<DagNodeLauncher,
  EventStandardMsg<DagNodeLauncher,
  DagNode::FINISH > >* event) {
  myOutput << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->nodeIsDone(event->getSource()->getNode()->getId().c_str(),
                           event->getSource()->getNode()->getDag()->
                           getWorkflow()->getId().c_str());
  }
}

void
WfLogDispatcher::onDagNodeFailed(
  const EventFrom<DagNodeLauncher,
  EventStandardMsg<DagNodeLauncher,
  DagNode::FAILED > >* event) {
  myOutput << *event << "\n";
  if (isOnline()) {
    myWfLogSrv->nodeIsFailed(event->getSource()->getNode()->getId().c_str(),
                             event->getSource()->getNode()->getDag()->
                             getWorkflow()->getId().c_str());
  }
}
