/* 
 * gpsk31  - PSK31 for Linux with a GTK+ Interface
 * 
 * Copyright (C) 2000 Luc Langehegermann, LX2GT
 * Copyright (C) 2008 Thomas Ries <tries@gmx.net>
 * Copyright (C) 2005,2006,2007,2008 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * The main author can be reached at pg4i@amsat.org or by smail-mail:
 * Joop Stakenborg, Bramengaarde 24, 3992KG Houten, The Netherlands.
 * 
 */

/*
 *  main.c - Main loop and initialisations
 */

#include <gtk/gtk.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <termios.h>
#include <errno.h>
#include "main_screen.h"

#include "misc.h"
#include "menu.h"
#include "spectrum.h"
#include "call_dialog.h"
#include "globals.h"
#include "phasescope.h"
#include "server.h"
#include "file.h"
#include "main.h"
#include "text.h"
#include "support.h"
#include "callback.h"
#include "socketif.h"

  #ifndef HAVE_LIBPTHREAD
  void master_handler(void);
  #endif

struct ini_settings ini_settings;	/* Initial Parameters */
extern int scconfig[];              /* [0]==8bit/16bit [1]==mono/stereo */
bool gpsk31debug = FALSE;
char configfile[128]="";		/* explicit config file to be loaded */

/* Our Main loop, called with gtk_timeout_add() */

gint
main_loop (gpointer data)
{
  static float rxfreqOld, rxfreq;	/* last rx freq */
  static int oldtx;			/* last tx/rx state */
  int phdelta;			/* last phase scope arc */
  int l;
  int sts;
  char str[2];
  GtkTextBuffer *buffer;
  GtkTextIter iter, end;

  #ifndef HAVE_LIBPTHREAD
   // 4 is an empirical value.
   // If choosen too low, sound card buffer will run empty, and thus sound
   // will have interruptions (fatal!). A value too high isn't that critical...
   for(int i=0; i<4; i++) { 
           master_handler();
  #endif



  /* Handle FFT */
  int N = getsamplecnt ();
  float fftval[N];
  l = commGetData (COMM_FFTCH, (char *) fftval, sizeof (fftval));
  if (l > 0)
    {
      spectrum_updateFFT (fftval, N);
    }


  #ifndef HAVE_LIBPTHREAD
   }
  #endif

  /* handle echo */

  char *time_string = get_time_string(NULL); /* Get the time string, must be freed later */

  char buf[256];
  l = commGetData (COMM_ECHOCH, buf, sizeof buf);
  if (l > 0)
    {
      for (int i = 0; i < l; i++)
	{
	  str[0] = buf[i];
	  str[1] = '\0';
	  if (str[0] == '\r')
	    break;
	  put_logfile (str); /* Save to the logfile if open */

	  if (str[0] == 0x7F || str[0] == '\b')
      {
        buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (main_screen.rxwindow));
        gtk_text_buffer_get_end_iter (buffer, &end);
        gtk_text_buffer_get_end_iter (buffer, &iter);
        gtk_text_iter_backward_chars (&iter, 1);
        gtk_text_buffer_delete (buffer, &iter, &end);
      }
	  else
	    put_rx_window (str, 1);
	}
    }
  /* Handle Receive */

  l = commGetData (COMM_RXCH, buf, sizeof buf);
  if (l > 0)
    {
      for (int i = 0; i < l; i++)
	{
	  str[0] = buf[i];
	  str[1] = '\0';
	  if (str[0] == '\r')
	    break;
	  put_logfile (str); /* Save to the logfile if open */

	  if (str[0] == 0x7F || str[0] == '\b')
      {
        buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (main_screen.rxwindow));
        gtk_text_buffer_get_end_iter (buffer, &end);
        gtk_text_buffer_get_end_iter (buffer, &iter);
        gtk_text_iter_backward_chars (&iter, 1);
        gtk_text_buffer_delete (buffer, &iter, &end);
      }
	  else {
	    put_rx_window (str, 0);
	    /* send to TCP connected client - if one is connected */
	    sts = tcp_send(str, strlen(str));
	  }
	}
    }

  PSK31info rxinfo;
  PSK31info txinfo;

  commGetInfo (COMM_RXCH, &rxinfo, sizeof (rxinfo));
  commGetInfo (COMM_TXCH, &txinfo, sizeof (txinfo));

  rxfreq = 0.01 * rxinfo.freq;
  phdelta = rxinfo.phdelta;

  /* Find out if we are TX or RX, and update the statusbar */

  if (rxinfo.ptt == 1 && oldtx == 0)
	{
        gtk_statusbar_pop (GTK_STATUSBAR (statusbar.txstate), 1);
	gtk_statusbar_push (GTK_STATUSBAR (statusbar.txstate),
                                          1, "TX");
	oldtx = 1;
        /* And write the info in the Log file */
	put_logfile ("\n\n<<<<<< Transmission started at: ");
        put_logfile (time_string);
        put_logfile (" >>>>>>\n\n");   
	}	
  else if (rxinfo.ptt == 0 && oldtx == 1) 
        {	
        gtk_statusbar_pop (GTK_STATUSBAR (statusbar.txstate), 1);
	gtk_statusbar_push (GTK_STATUSBAR (statusbar.txstate),
                                          1, "RX");
	oldtx = 0;
	/* And write the info to the Log file */
	put_logfile ("\n\n<<<<<< Transmission stopped at: ");
        put_logfile (time_string);
        put_logfile (" >>>>>>\n\n");
	}

  if (rxinfo.ptt == 0)
    {
      /* Receiving...... */
	draw_scope (rxinfo.phdelta, rxinfo.strength, rxinfo.dcd);
      /* Update RX freq spin button */
      if (!GTK_WIDGET_HAS_FOCUS (main_screen.rx_freq) &&
	  !GTK_WIDGET_HAS_FOCUS (main_screen.tx_freq))
	{
	  if ((rxfreq - rxfreqOld) > .05 || rxfreqOld - rxfreq > .05)
	    {
	      gtk_spin_button_set_value (GTK_SPIN_BUTTON
					 (main_screen.rx_freq), rxfreq);
	      rxfreqOld = rxfreq;
	    }
	}
    }
  g_free (time_string);

  /* check the TCP connection for received characters and feed them into
     GPSK31 */
  sts = tcp_read(buf, sizeof(buf)-1);
  /* sts>0 -> received something? */
  if (sts > 0) {
    /* feed into gpsk31 like sending a Fx-key macro (macro processing) */
    macro_processor2(buf);
  }

  /* check for a new TCP connection */
  tcp_connect();

  return (true);
}

static void
parsecommandline (int argc, char *argv[])
{
	extern char *optarg;	/* Defined in libc getopt and unistd.h */
	gint p;
	int i;

	while ((p = getopt (argc, argv, "dhvc:")) != -1)
	{
		switch (p)
		{
            case ':':
            case '?':
            case 'h':
            	g_print ("Usage: %s [option]\n", PACKAGE);
            	g_print ("    -d        Enable debugging\n");
            	g_print ("    -h        Display this help and exit\n");
            	g_print ("    -v        Output version information and exit\n");
            	g_print ("    -c <file> Load the specific config <file>\n");
            	exit (0);
            break;
            case 'c':
            	i=sizeof(configfile)-1;
            	strncpy(configfile,optarg,i);
            	configfile[i]='\0';
            break;
            case 'd':
            	gpsk31debug = TRUE;
            break;
            case 'v':
            	gchar *versionstr = g_strdup_printf ("%s version %s\n",
            	g_path_get_basename (argv[0]), VERSION);
            	g_print (versionstr);
            	g_free (versionstr);
            	exit (0);
            break;
		}
	}
}

int
main (int argc, char *argv[])
{

  /* create .gpsk31 directory */

  char *home = getenv("HOME");
  char direct[strlen (home) + 8];
  char fn[strlen(home) + 50];
  strcpy (direct, home);
  strcat (direct, "/.gpsk31");
  mkdir (direct, 00700);
  strcpy (fn,direct);
  strcat (fn, "/");

  /* sound card config */
  scconfig[0] = 1;
  scconfig[1] = 1;
  
  parsecommandline (argc, argv);
  gtk_init (&argc, &argv);      /* init gtk+ */

  add_pixmap_directory (PIXMAPDIR);
  read_config ();		/* read gpsk.conf and the ini_settings structure */
  setup_screen ();		/* build the main window */
  set_config ();		/* set the configuration */
  init_call_dialog ();          /* init the logbook dialog */
  
  /* allow full path, starting with '/' */
  if (ini_settings.log_filename[0] == '/' ) {
    strcpy(fn,ini_settings.log_filename);
  } else {
    strcat(fn,ini_settings.log_filename);
  }
  logfile(fn);

  /* These have always the same values when we start */
  commControl (COMM_TXCH, COMM_QPSK, 0);
  commControl (COMM_RXCH, COMM_QPSK, 0);
  commControl (COMM_RXCH, COMM_AFC, 1);
  check_menu ("/MainMenu/ModeMenu/AFC", 1);
  check_menu ("/MainMenu/ModeMenu/NET", 1);

  g_signal_connect (G_OBJECT (main_screen.window), "delete_event",
		      G_CALLBACK (exit_dialog), NULL);
  g_timeout_add (250, display_time, NULL);	/* timeout for clock */

  #ifndef HAVE_LIBPTHREAD
  gtk_idle_add (main_loop, NULL);
  #else
  g_timeout_add (50, main_loop, NULL);
  #endif

  /* post listen TCP socket for remote operation */
  tcp_set_listen_port(ini_settings.tcp_listen_port);

  /* go into gtk_main loop and never come back */
  gtk_main ();
}
