# JerryScript debugger transport interface

The transport interface support allows dynamic selection of transportation
layers which can encode/decode or send/receive messages transmitted between
the debugger client and server.

# Types

## jerry_debugger_transport_receive_context_t

**Summary**

This context represents the current status of processing received data.
The final state is returned by
[jerry_debugger_transport_receive](#jerry_debugger_transport_receive)
and must be passed to
[jerry_debugger_transport_receive_completed](#jerry_debugger_transport_receive_completed)
after the message is processed.

**Prototype**

```c
typedef struct
{
  uint8_t *buffer_p; /**< buffer for storing the received data */
  size_t received_length; /**< number of currently received bytes */
  uint8_t *message_p; /**< start of the received message */
  size_t message_length; /**< length of the received message */
  size_t message_total_length; /**< total length for datagram protocols,
                                *   0 for stream protocols */
} jerry_debugger_transport_receive_context_t;
```

## jerry_debugger_transport_header_t

**Summary**

Shared header for each transport interface. It mostly contains callback functions
used by the JerryScript debugger server.

**Prototype**

```c
typedef struct jerry_debugger_transport_layer_t
{
  /* The following fields must be filled before calling jerry_debugger_transport_add(). */
  jerry_debugger_transport_close_t close; /**< close connection callback */
  jerry_debugger_transport_send_t send;  /**< send data callback */
  jerry_debugger_transport_receive_t receive; /**< receive data callback */

  /* The following fields are filled by jerry_debugger_transport_add(). */
  struct jerry_debugger_transport_layer_t *next_p; /**< next transport layer */
} jerry_debugger_transport_header_t;
```

## jerry_debugger_transport_close_t

**Summary**

Called when the connection is closed. Must release all resources (including the
memory area for the transport interface) allocated for the transport interface.

**Prototype**

```c
typedef void (*jerry_debugger_transport_close_t) (struct jerry_debugger_transport_interface_t *header_p);
```

## jerry_debugger_transport_send_t

**Summary**

Called when a message needs to be sent. Must either transmit the message or call
the `header_p->next_p->send()` method.

**Prototype**

```c
typedef bool (*jerry_debugger_transport_send_t) (struct jerry_debugger_transport_interface_t *header_p,
                                                 uint8_t *message_p, size_t message_length);
```

## jerry_debugger_transport_receive_t

**Summary**

Called during message processing. If messages are available it must return with
the next message.

**Prototype**

```c
typedef bool (*jerry_debugger_transport_receive_t) (struct jerry_debugger_transport_interface_t *header_p,
                                                    jerry_debugger_transport_receive_context_t *context_p);
```

# Transport interface API functions

## jerry_debugger_transport_add

**Summary**

Add a new interface to the transporation interface chain. The interface
will be the first item of the interface chain.

**Prototype**

```c
void jerry_debugger_transport_add (jerry_debugger_transport_header_t *header_p,
                                   size_t send_message_header_size, size_t max_send_message_size,
                                   size_t receive_message_header_size, size_t max_receive_message_size);
```

- `header_p`: header of a transporation interface.
- `send_message_header_size`: size of the outgoing message header, can be 0.
- `max_send_message_size`: maximum outgoing message size supported by the interface.
- `receive_message_header_size`: size of the incoming message header, can be 0.
- `max_receive_message_size`: maximum incoming message size supported by the interface.

## jerry_debugger_transport_start

**Summary**

Starts the communication to the debugger client. Must be called after the
connection is successfully established.

**Prototype**

```c
void jerry_debugger_transport_start (void);
```

## jerry_debugger_transport_is_connected

**Summary**

Tells whether a debugger client is connected to the debugger server.

**Prototype**

```c
bool jerry_debugger_transport_is_connected (void);
```

- return value: `true`, if a client is connected, `false` otherwise.

## jerry_debugger_transport_close

**Summary**

Disconnect from the current debugger client. It does nothing if a client is
not connected.

**Prototype**

```c
void jerry_debugger_transport_close (void);
```

## jerry_debugger_transport_send

**Summary**

Send message to the client.

**Prototype**

```c
bool jerry_debugger_transport_send (const uint8_t *message_p, size_t message_length);
```

- `message_p`: message to be sent.
- `message_length`: message length in bytes.
- return value: `true`, if a client is still connected, `false` otherwise.

## jerry_debugger_transport_receive

**Summary**

Receive message from the client.

**Prototype**

```c
bool jerry_debugger_transport_receive (jerry_debugger_transport_receive_context_t *context_p);
```

- `context_p`: an unused [jerry_debugger_transport_receive_context_t](#jerry_debugger_transport_receive_context_t).
- return value: `true`, if a client is still connected, `false` otherwise.

## jerry_debugger_transport_receive_completed

**Summary**

Must be called after [jerry_debugger_transport_receive](#jerry_debugger_transport_receive)
returns with a valid message. Must not be called otherwise.

**Prototype**

```c
void jerry_debugger_transport_receive_completed (jerry_debugger_transport_receive_context_t *context_p);
```

- `context_p`: a [jerry_debugger_transport_receive_context_t](#jerry_debugger_transport_receive_context_t)
               passed to [jerry_debugger_transport_receive](#jerry_debugger_transport_receive).

## jerry_debugger_transport_sleep

**Summary**

Can be used to wait for incoming messages. Currently the delay is 100ms.

**Prototype**

```c
void jerry_debugger_transport_sleep (void);
```
