/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Wrapper_p.h"

#include <map>
#include <vector>

#include <llvm/DerivedTypes.h>
#include <llvm/Function.h>
#include <llvm/Module.h>

#include "GTLCore/Function_p.h"
#include "GTLCore/Macros.h"
#include "GTLCore/Macros_p.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/Parameter.h"
#include "GTLCore/PixelDescription.h"
#include "GTLCore/Type.h"
#include "GTLCore/Type_p.h"
#include "GTLCore/TypesManager.h"
#include "GTLCore/TypesManager_p.h"
#include "GTLCore/Value.h"
#include "GTLCore/VirtualMachine_p.h"

#include "GTLCore/AbstractImage.h"
#include "CodeGenerator_p.h"
#include "Debug.h"
#include "PixelVisitor_p.h"
#include "PixelConvertExpressionFactory_p.h"

#include "wrappers/ImageWrap_p.h"
#include "LibrariesManager.h"
#include <GTLCore/CompilationMessages.h>

using namespace OpenShiva;

struct WrappedFunctions {
  void* memToVec;
  void* vecToMem;
};

typedef std::map< GTLCore::PixelDescription, WrappedFunctions > pdToWF;
typedef pdToWF::iterator pdToWF_it;
typedef pdToWF::const_iterator pdToWF_cit;

struct Wrapper::Private {
  pdToWF imageFunctions;
  Kernel* kernel;
  bool loadShivaWrapper;
};

Wrapper::Wrapper(Kernel* _kernel, GTLCore::ModuleData* _moduleData, llvm::Module* module, int channels, bool loadShivaWrapper) : GTLFragment::Wrapper(_moduleData, module, channels), d(new Private)
{
  d->loadShivaWrapper = loadShivaWrapper;
  GTL_ASSERT( channels > 0 );
  d->kernel = _kernel;
}

Wrapper::~Wrapper()
{
  delete d;
}

ImageWrap* Wrapper::wrapImage(GTLCore::AbstractImage* _abstractImage)
{
  SHIVA_DEBUG("wrapImage");
  ImageWrap* owrap = new ImageWrap;
  owrap->image = _abstractImage;
  pdToWF_it it = d->imageFunctions.find( _abstractImage->pixelDescription() );
  if( it == d->imageFunctions.end() )
  {
    // Wrap functions doesn't exist yet, generate them
    WrappedFunctions wf;
    wf.memToVec = GTLCore::VirtualMachine::instance()->getPointerToFunction(
        CodeGenerator::generateMemToVec( moduleData(), llvmModule(), _abstractImage->pixelDescription()  ) );
    wf.vecToMem = GTLCore::VirtualMachine::instance()->getPointerToFunction(
        CodeGenerator::generateVecToMem( moduleData(), llvmModule(), _abstractImage->pixelDescription()  ) );
    d->imageFunctions[ _abstractImage->pixelDescription() ] = wf;
    owrap->memToVec = wf.memToVec;
    owrap->vecToMem = wf.vecToMem;
  } else {
    // Use already existing wrap function
    owrap->memToVec = it->second.memToVec;
    owrap->vecToMem = it->second.vecToMem;
  }
  SHIVA_DEBUG("Image wrapped");
  return owrap;
}

void Wrapper::fillTypesManager( GTLCore::TypesManager* _typesManager, GTLCore::ConvertCenter* _convertCenter )
{
  SHIVA_DEBUG("fillTypesManager");
  /*const GTLCore::Type* colorType = */createColorType(_typesManager, _convertCenter);
  _convertCenter->addConvertExpressionFactory(new PixelConvertExpressionFactory);
  for( int i = 1; i <= 5; ++i)
  {
    // Create Image type
    const GTLCore::Type* pixelType = createPixelType( moduleData(), llvmModule(), _typesManager, _convertCenter, GTLCore::String::number( i ), i );
    createImageType( moduleData(), llvmModule(), _typesManager, GTLCore::String::number( i ), i, pixelType );
  }
  const GTLCore::Type* pixelType = createPixelType( moduleData(), llvmModule(), _typesManager, _convertCenter, "", channels() );
  createImageType( moduleData(), llvmModule(), _typesManager, "", channels(), pixelType );
  const GTLCore::Type* regionType = createRegionType( moduleData(), llvmModule(), _typesManager );

  if(d->loadShivaWrapper)
  {
    GTLFragment::Library* library = OpenShiva::LibrariesManager::instance()->loadLibrary( "shivawrappers", channels());
    GTL_ASSERT( library );
    if(not library->isCompiled())
    {
      library->compile();
      if(not library->isCompiled())
      {
        GTL_ABORT("shivawrappers.shiva compilation failed, " << library->compilationMessages().toString());
        return;
      }
    }
    moduleData()->linkWith( library->data() );
    addFunctionFromModuleToType(regionType, library, GTLCore::ScopedName("wrappers", "translate"));
    addFunctionFromModuleToType(regionType, library, GTLCore::ScopedName("wrappers", "translated"));
    addFunctionFromModuleToType(regionType, library, GTLCore::ScopedName("wrappers", "topLeft"));
    addFunctionFromModuleToType(regionType, library, GTLCore::ScopedName("wrappers", "bottomRight"));
  }
}


llvm::FunctionType* Wrapper::pixel_wrap_set_alpha_type(llvm::LLVMContext& context, GTLCore::TypesManager* _typesManager, const GTLCore::Type* _pixelType)
{
  std::vector<llvm::Type*> functionArgs;
  functionArgs.push_back( _pixelType->d->pointerType(context) );
  functionArgs.push_back( llvm::Type::getFloatTy(context) );
  return llvm::FunctionType::get( llvm::Type::getVoidTy(context), functionArgs, false);
}

llvm::FunctionType* Wrapper::pixel_wrap_alpha_type( llvm::LLVMContext& context, GTLCore::TypesManager* _typesManager, const GTLCore::Type* _pixelType)
{
  std::vector<llvm::Type*> functionArgs;
  functionArgs.push_back( _pixelType->d->pointerType(context) );
  return llvm::FunctionType::get( llvm::Type::getFloatTy(context), functionArgs, false);
}

const GTLCore::Type* Wrapper::createPixelType( GTLCore::ModuleData* _module, llvm::Module* llvmModule, GTLCore::TypesManager* _typesManager, GTLCore::ConvertCenter* _convertCenter, const GTLCore::String& _suffix, int _channels )
{
  //---------------------- WARNING ----------------------//
  // Whenever the following structure is edited,         //
  // its C++ declaration must be changed too in          //
  // struct PixelWrap !                                  //
  //---------------------- WARNING ----------------------//
  
  llvm::LLVMContext& context = _module->llvmModule()->getContext();
  
  const GTLCore::Type* vecType = vectorType( _typesManager, _channels );
  std::vector<GTLCore::Type::StructDataMember> pixelDataMembers;
  pixelDataMembers.push_back( GTLCore::Type::StructDataMember( "", GTLCore::Type::Pointer ) ); // AbstractColorConverter*
  pixelDataMembers.push_back( GTLCore::Type::StructDataMember( "", GTLCore::Type::Integer32 ) ); // alphapos
  
  pixelDataMembers.push_back( GTLCore::Type::StructDataMember( "coord", _typesManager->getVector( GTLCore::Type::Float32, 2 ) ) );
  
  pixelDataMembers.push_back( GTLCore::Type::StructDataMember( "data", vecType ) );
  
  
  const GTLCore::Type* type = _typesManager->d->createStructure( "pixel" + _suffix, pixelDataMembers );
  type->d->setVisitor( PixelVisitor::instance() );
  
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createInternalFunction(
                _module, context, "setOpacity",
                CodeGenerator::generatePixelSetAlpha( _module, llvmModule, type, _channels ),
                GTLCore::Type::Void, GTLCore::Function::Private::EFP_ONLY_TYPE, 2,
                type, GTLCore::Type::Float32 ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createInternalFunction(
                _module, context, "opacity",
                CodeGenerator::generatePixelAlpha( _module, llvmModule, type, _channels ),
                GTLCore::Type::Float32, GTLCore::Function::Private::EFP_ONLY_TYPE, 1,
                type ) ) );

  _convertCenter->addAutoConversion( type, vecType );
  
  return type;
}

const GTLCore::Type* Wrapper::createRegionType( GTLCore::ModuleData* _module, llvm::Module* _llvmModule, GTLCore::TypesManager* _typesManager )
{
  //---------------------- WARNING ----------------------//
  // Whenever the following structure is edited,         //
  // its C++ declaration must be changed too in          //
  // struct RegionWrap !                                 //
  //---------------------- WARNING ----------------------//
  
  llvm::LLVMContext& context = _module->llvmModule()->getContext();
  
  std::vector<GTLCore::Type::StructDataMember> regionDataMembers;
  regionDataMembers.push_back( GTLCore::Type::StructDataMember( "x", GTLCore::Type::Float32 ) );
  regionDataMembers.push_back( GTLCore::Type::StructDataMember( "y", GTLCore::Type::Float32) );
  regionDataMembers.push_back( GTLCore::Type::StructDataMember( "columns", GTLCore::Type::Float32 ) );
  regionDataMembers.push_back( GTLCore::Type::StructDataMember( "rows", GTLCore::Type::Float32 ) );
  const GTLCore::Type* type = _typesManager->d->createStructure( "region", regionDataMembers);
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, _llvmModule, context, "left", "region_wrap_left", GTLCore::Type::Float32,
                GTLCore::Function::Private::EFP_ONLY_TYPE, 1,
                type ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, _llvmModule, context, "right", "region_wrap_right", GTLCore::Type::Float32,
                GTLCore::Function::Private::EFP_ONLY_TYPE, 1,
                type ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, _llvmModule, context, "bottom", "region_wrap_bottom", GTLCore::Type::Float32,
                GTLCore::Function::Private::EFP_ONLY_TYPE, 1,
                type ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, _llvmModule, context, "top", "region_wrap_top", GTLCore::Type::Float32,
                GTLCore::Function::Private::EFP_ONLY_TYPE, 1,
                type ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, _llvmModule, context, "intersect", "region_wrap_intersect", GTLCore::Type::Void,
                GTLCore::Function::Private::EFP_ONLY_TYPE, 2,
                type, type ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, _llvmModule, context, "union", "region_wrap_union", GTLCore::Type::Void,
                GTLCore::Function::Private::EFP_ONLY_TYPE, 2,
                type, type ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, _llvmModule, context, "outset", "region_wrap_outset", GTLCore::Type::Void,
                GTLCore::Function::Private::EFP_ONLY_TYPE, 2,
                type, GTLCore::Type::Float32 ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, _llvmModule, context, "inset", "region_wrap_inset", GTLCore::Type::Void,
                GTLCore::Function::Private::EFP_ONLY_TYPE, 2,
                type, GTLCore::Type::Float32 ) ) );
  return type;
}

llvm::Function* Wrapper::image_wrap_dataFunction( llvm::Module* _module, const GTLCore::Type* _imageType )
{
  GTL_ASSERT( _imageType );
  
  llvm::LLVMContext& context = _module->getContext();
  
  std::vector<llvm::Type*> functionArgs;
  functionArgs.push_back(llvm::PointerType::get( _imageType->d->type(context), 0));
  functionArgs.push_back(llvm::Type::getInt32Ty(context));
  functionArgs.push_back(llvm::Type::getInt32Ty(context));
  llvm::FunctionType* functionTy = llvm::FunctionType::get( llvm::PointerType::get(llvm::Type::getInt8Ty(context), 0), functionArgs, false);
  return (llvm::Function*)_module->getOrInsertFunction( "image_wrap_data", functionTy);
}

llvm::Function* Wrapper::image_wrap_const_dataFunction( llvm::Module* _module, const GTLCore::Type* _imageType )
{
  GTL_ASSERT( _imageType );
  std::vector<llvm::Type*> functionArgs;
  functionArgs.push_back(llvm::PointerType::get( _imageType->d->type(_module->getContext()), 0));
  functionArgs.push_back(llvm::IntegerType::get(_module->getContext(), 32));
  functionArgs.push_back(llvm::IntegerType::get(_module->getContext(), 32));
  llvm::FunctionType* functionTy = llvm::FunctionType::get( llvm::PointerType::get(llvm::IntegerType::get(_module->getContext(), 8), 0), functionArgs, false);
  return (llvm::Function*)_module->getOrInsertFunction( "image_wrap_const_data", functionTy);
}


llvm::Function* Wrapper::image_alpha_posFunction( llvm::Module* _module, const GTLCore::Type* _imageType )
{
  GTL_ASSERT( _imageType );
  
  llvm::LLVMContext& context = _module->getContext();
  
  std::vector<llvm::Type*> functionArgs;
  functionArgs.push_back(llvm::PointerType::get( _imageType->d->type(context), 0));
  llvm::FunctionType* functionTy = llvm::FunctionType::get(llvm::IntegerType::getInt32Ty(context), functionArgs, false);
  return (llvm::Function*)_module->getOrInsertFunction( "image_alpha_pos", functionTy);
}

llvm::Function* Wrapper::image_color_converter( llvm::Module* _module, const GTLCore::Type* _imageType )
{
  GTL_ASSERT( _imageType );
  
  llvm::LLVMContext& context = _module->getContext();
  
  std::vector<llvm::Type*> functionArgs;
  functionArgs.push_back(llvm::PointerType::get( _imageType->d->type(context), 0));
  llvm::FunctionType* functionTy = llvm::FunctionType::get(GTLCore::Type::Pointer->d->type(_module->getContext()), functionArgs, false);
  return (llvm::Function*)_module->getOrInsertFunction( "image_color_converter", functionTy);
}

llvm::FunctionType* Wrapper::image_wrap_sample_nearest_type( llvm::LLVMContext& context, GTLCore::TypesManager* _typesManager, const GTLCore::Type* _imageType, const GTLCore::Type* _pixelType )
{
  std::vector<llvm::Type*> functionArgs;
  functionArgs.push_back( _imageType->d->pointerType(context) );
  functionArgs.push_back( _typesManager->getVector( GTLCore::Type::Float32, 2 )->d->type(context) );
  return llvm::FunctionType::get( _pixelType->d->pointerType(context) , functionArgs, false);
}

llvm::FunctionType* Wrapper::image_wrap_mem_to_vec_float_type( llvm::LLVMContext& context, GTLCore::TypesManager* _typesManager, int _channels )
{
  std::vector<llvm::Type*> functionArgs;
  functionArgs.push_back( llvm::PointerType::get( vectorType( _typesManager, _channels)->d->type(context), 0) );
  functionArgs.push_back( llvm::PointerType::get( llvm::Type::getInt8Ty(context), 0));
  return llvm::FunctionType::get( llvm::Type::getVoidTy(context), functionArgs, false);
}

llvm::FunctionType* Wrapper::image_wrap_vec_float_to_mem_type( llvm::LLVMContext& context, GTLCore::TypesManager* _typesManager, int _channels )
{
  std::vector<llvm::Type*> functionArgs;
  functionArgs.push_back( llvm::PointerType::get( llvm::IntegerType::getInt8Ty(context), 0));
  functionArgs.push_back( llvm::PointerType::get( vectorType( _typesManager, _channels)->d->type(context), 0) );
  functionArgs.push_back( llvm::Type::getInt64Ty(context) ); // gtl_uint64 _channelFlags
  return llvm::FunctionType::get( llvm::Type::getVoidTy(context), functionArgs, false);
}

class Wrapper::Mem2VecFloatTypeAribtraryTypeFactory : public GTLCore::Type::Private::AribtraryTypeFactory {
  public:
    Mem2VecFloatTypeAribtraryTypeFactory( GTLCore::TypesManager* _typesManager, int _channels, llvm::LLVMContext& _context) : m_typesManager(_typesManager), m_channels(_channels), m_context(_context)
    {
    }
    virtual llvm::Type* createType(llvm::LLVMContext& context)
    {
      return llvm::PointerType::get( Wrapper::image_wrap_mem_to_vec_float_type( m_context, m_typesManager, m_channels ), 0 );
    }
  private:
    GTLCore::TypesManager* m_typesManager;
    int m_channels;
    llvm::LLVMContext& m_context;
};

class Wrapper::VecFloat2MemTypeAribtraryTypeFactory : public GTLCore::Type::Private::AribtraryTypeFactory {
  public:
    VecFloat2MemTypeAribtraryTypeFactory( GTLCore::TypesManager* _typesManager, int _channels, llvm::LLVMContext& _context) : m_typesManager(_typesManager), m_channels(_channels), m_context(_context)
    {
    }
    virtual llvm::Type* createType(llvm::LLVMContext& context)
    {
      return llvm::PointerType::get( Wrapper::image_wrap_vec_float_to_mem_type( m_context, m_typesManager, m_channels ), 0 );
    }
  private:
    GTLCore::TypesManager* m_typesManager;
    int m_channels;
    llvm::LLVMContext& m_context;
};

void Wrapper::createImageType( GTLCore::ModuleData* _moduleData, llvm::Module* _llvmModule, GTLCore::TypesManager* _typesManager, const GTLCore::String& _suffix, int _channels, const GTLCore::Type* _pixelType )
{
  //---------------------- WARNING ----------------------//
  // Whenever the following structure is edited,         //
  // its C++ declaration must be changed too in          //
  // struct ImageWrap !                                  //
  //---------------------- WARNING ----------------------//
  std::vector<GTLCore::Type::StructDataMember> imageDataMembers;
  imageDataMembers.push_back( GTLCore::Type::StructDataMember( "image", GTLCore::Type::Pointer) );
  imageDataMembers.push_back( GTLCore::Type::StructDataMember(
        "memToVec",
        GTLCore::Type::Private::createArbitraryType( new Mem2VecFloatTypeAribtraryTypeFactory( _typesManager, _channels, _moduleData->llvmModule()->getContext() ) ) ) ); // FIXME arbitraty type are leaking
  imageDataMembers.push_back( GTLCore::Type::StructDataMember( 
        "vecToMem",
        GTLCore::Type::Private::createArbitraryType( new VecFloat2MemTypeAribtraryTypeFactory( _typesManager, _channels, _moduleData->llvmModule()->getContext() ) ) ) ); // FIXME arbitraty type are leaking
  const GTLCore::Type* type = _typesManager->d->createStructure( "image" + _suffix, imageDataMembers );
  
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createInternalFunction(
                _moduleData, _moduleData->llvmModule()->getContext(), "sampleNearest",
                CodeGenerator::generateImageSampleNearest( _moduleData, _llvmModule, type, _pixelType ),
                _pixelType, GTLCore::Function::Private::EFP_ONLY_TYPE, 2,
                type, _moduleData->typesManager()->getVector( GTLCore::Type::Float32, 2 ) ) ) );
}

const GTLCore::Type* Wrapper::vectorType( GTLCore::TypesManager* _typesManager, int _channels )
{
  if( _channels == 1)
  {
    return GTLCore::Type::Float32;
  }
  return _typesManager->getVector( GTLCore::Type::Float32, _channels);
}
