///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/plugins/NativePluginClassDescriptor.h>
#include <core/plugins/PluginClass.h>
#include <core/plugins/Plugin.h>
#include <core/plugins/PluginManager.h>
#include <core/reference/PropertyFieldDescriptor.h>

namespace Core {

NativePluginClassInfo* NativePluginClassInfo::_firstInfo = NULL;

/// The descriptor of the root class of all plugin classes.
/// This class is named "PluginClass".
NativePluginClassDescriptor NativePluginClassDescriptor::nativeRootClass;

/******************************************************************************
* Constructor of the descriptor structure that initializes the root element.
******************************************************************************/
NativePluginClassDescriptor::NativePluginClassDescriptor() :
	PluginClassDescriptor("PluginClass", NULL, NULL, QDomElement(), true, true),
	classInfo(NULL)
{
}

/******************************************************************************
* Constructor of the descriptor structure.
******************************************************************************/
NativePluginClassDescriptor::NativePluginClassDescriptor(const QString& name, PluginClassDescriptor* baseClass, Plugin* plugin, const QDomElement& classNode, bool isAbstract, bool isSerializable) :
	PluginClassDescriptor(name, baseClass, plugin, classNode, isAbstract, isSerializable),
	classInfo(NULL)
{
}

/******************************************************************************
* Creates an object of the appropriate kind.
* Throws an exception if the containing plugin failed to load.
******************************************************************************/
intrusive_ptr<PluginClass> NativePluginClassDescriptor::createInstanceImpl(bool isLoading)
{
	CHECK_POINTER(classInfo);

	if(classInfo->singletonInstance != NULL) {
		OVITO_ASSERT_MSG(!isSerializable(), "PluginClassDescriptor::createInstance", "A singleton plugin class must not be serializable.");
#ifdef _DEBUG
		// Make sure that the author has overidden the virtual autoDeleteObject() function.
		classInfo->singletonInstance->autoDeleteObject();
		// The object should be still there...
		CHECK_OBJECT_POINTER(classInfo->singletonInstance);
#endif
		return classInfo->singletonInstance;
	}

	UndoSuspender noUndo;

	PluginClass* obj;
	if(isSerializable())
		obj = classInfo->createInstance(isLoading);
	else
		obj = classInfo->createInstance();

	if(obj == NULL)
		throw Exception(Plugin::tr("Cannot instantiate abstract plugin class '%1'.").arg(name()));

	return obj;
}

/******************************************************************************
* Associates this plugin class descriptor with the corresponding
* class info record after a native plugin has been loaded.
******************************************************************************/
void NativePluginClassDescriptor::associateWithClassInfo(NativePluginClassInfo* classInfo)
{
	// Create a bi-directional association.
	this->classInfo = classInfo;
	this->_firstPropertyField = classInfo->firstNativePropertyField();
	classInfo->_descriptor = this;
}

/******************************************************************************
* Searches for a property field defined in this class.
******************************************************************************/
const PropertyFieldDescriptor* NativePluginClassInfo::findNativePropertyField(const char* identifier) const
{
	for(const PropertyFieldDescriptor* field = firstNativePropertyField(); field; field = field->next())
		if(qstrcmp(field->identifier(), identifier) == 0) return field;
	return NULL;
}


};
