"""Tests for dynamical matrix classes."""

from __future__ import annotations

import numpy as np
import pytest

from phonopy import Phonopy
from phonopy.harmonic.dynamical_matrix import DynamicalMatrix, DynamicalMatrixGL

dynmat_ref_000 = [
    0.052897,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    -0.042597,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.052897,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    -0.042597,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.052897,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    -0.042597,
    0.000000,
    -0.042597,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.034302,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    -0.042597,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.034302,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    -0.042597,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.034302,
    0.000000,
]


dynmat_ref_252525 = [
    0.075295,
    0.000000,
    0.016777,
    0.000000,
    0.016777,
    0.000000,
    -0.040182,
    0.000000,
    -0.004226,
    0.000000,
    -0.004226,
    0.000000,
    0.016777,
    0.000000,
    0.075295,
    0.000000,
    0.016777,
    0.000000,
    -0.004226,
    0.000000,
    -0.040182,
    0.000000,
    -0.004226,
    0.000000,
    0.016777,
    0.000000,
    0.016777,
    0.000000,
    0.075295,
    0.000000,
    -0.004226,
    0.000000,
    -0.004226,
    0.000000,
    -0.040182,
    0.000000,
    -0.040182,
    0.000000,
    -0.004226,
    0.000000,
    -0.004226,
    0.000000,
    0.055704,
    0.000000,
    0.011621,
    0.000000,
    0.011621,
    0.000000,
    -0.004226,
    0.000000,
    -0.040182,
    0.000000,
    -0.004226,
    0.000000,
    0.011621,
    0.000000,
    0.055704,
    0.000000,
    0.011621,
    0.000000,
    -0.004226,
    0.000000,
    -0.004226,
    0.000000,
    -0.040182,
    0.000000,
    0.011621,
    0.000000,
    0.011621,
    0.000000,
    0.055704,
    0.000000,
]

dynmat_gonze_lee_ref_252525 = [
    0.081339,
    0.000000,
    0.029509,
    0.000000,
    0.029509,
    0.000000,
    -0.045098,
    0.000000,
    -0.015204,
    0.000000,
    -0.015204,
    0.000000,
    0.029509,
    0.000000,
    0.081339,
    0.000000,
    0.029509,
    0.000000,
    -0.015204,
    0.000000,
    -0.045098,
    0.000000,
    -0.015204,
    0.000000,
    0.029509,
    0.000000,
    0.029509,
    0.000000,
    0.081339,
    0.000000,
    -0.015204,
    0.000000,
    -0.015204,
    0.000000,
    -0.045098,
    0.000000,
    -0.045098,
    0.000000,
    -0.015204,
    0.000000,
    -0.015204,
    0.000000,
    0.059623,
    0.000000,
    0.019878,
    0.000000,
    0.019878,
    0.000000,
    -0.015204,
    0.000000,
    -0.045098,
    0.000000,
    -0.015204,
    0.000000,
    0.019878,
    0.000000,
    0.059623,
    0.000000,
    0.019878,
    0.000000,
    -0.015204,
    0.000000,
    -0.015204,
    0.000000,
    -0.045098,
    0.000000,
    0.019878,
    0.000000,
    0.019878,
    0.000000,
    0.059623,
    0.000000,
]

dynmat_gonze_lee_full_ref_252525 = [
    0.076944,
    0.000000,
    0.020251,
    -0.000000,
    0.020251,
    0.000000,
    -0.041523,
    -0.000000,
    -0.007221,
    0.000000,
    -0.007221,
    0.000000,
    0.020251,
    0.000000,
    0.076944,
    0.000000,
    0.020251,
    0.000000,
    -0.007221,
    0.000000,
    -0.041523,
    -0.000000,
    -0.007221,
    0.000000,
    0.020251,
    -0.000000,
    0.020251,
    -0.000000,
    0.076944,
    0.000000,
    -0.007221,
    0.000000,
    -0.007221,
    0.000000,
    -0.041523,
    -0.000000,
    -0.041523,
    0.000000,
    -0.007221,
    -0.000000,
    -0.007221,
    -0.000000,
    0.056774,
    0.000000,
    0.013874,
    -0.000000,
    0.013874,
    0.000000,
    -0.007221,
    -0.000000,
    -0.041523,
    0.000000,
    -0.007221,
    -0.000000,
    0.013874,
    0.000000,
    0.056774,
    0.000000,
    0.013874,
    -0.000000,
    -0.007221,
    -0.000000,
    -0.007221,
    -0.000000,
    -0.041523,
    0.000000,
    0.013874,
    -0.000000,
    0.013874,
    0.000000,
    0.056774,
    0.000000,
]

dynmat_wang_ref_252525 = [
    0.081339,
    -0.000000,
    0.022821,
    0.000000,
    0.022821,
    0.000000,
    -0.045098,
    -0.000000,
    -0.009142,
    -0.000000,
    -0.009142,
    0.000000,
    0.022821,
    0.000000,
    0.081339,
    0.000000,
    0.022821,
    0.000000,
    -0.009142,
    0.000000,
    -0.045098,
    0.000000,
    -0.009142,
    0.000000,
    0.022821,
    0.000000,
    0.022821,
    0.000000,
    0.081339,
    0.000000,
    -0.009142,
    0.000000,
    -0.009142,
    0.000000,
    -0.045098,
    0.000000,
    -0.045098,
    0.000000,
    -0.009142,
    0.000000,
    -0.009142,
    0.000000,
    0.059623,
    0.000000,
    0.015541,
    0.000000,
    0.015541,
    0.000000,
    -0.009142,
    0.000000,
    -0.045098,
    0.000000,
    -0.009142,
    0.000000,
    0.015541,
    0.000000,
    0.059623,
    0.000000,
    0.015541,
    0.000000,
    -0.009142,
    0.000000,
    -0.009142,
    0.000000,
    -0.045098,
    0.000000,
    0.015541,
    0.000000,
    0.015541,
    -0.000000,
    0.059623,
    0.000000,
]

dynmat_ref_555 = [
    0.091690,
    0.000000,
    0.033857,
    0.000000,
    0.033857,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.033857,
    0.000000,
    0.091690,
    0.000000,
    0.033857,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.033857,
    0.000000,
    0.033857,
    0.000000,
    0.091690,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.064909,
    0.000000,
    0.021086,
    0.000000,
    0.021086,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.021086,
    0.000000,
    0.064909,
    0.000000,
    0.021086,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.000000,
    0.021086,
    0.000000,
    0.021086,
    0.000000,
    0.064909,
    0.000000,
]


@pytest.mark.parametrize(
    "is_compact_fc,lang", [(True, "C"), (False, "C"), (True, "Py"), (False, "Py")]
)
def test_dynmat(
    ph_nacl_nonac: Phonopy,
    ph_nacl_nonac_compact_fc: Phonopy,
    is_compact_fc: bool,
    lang: str,
):
    """Test dynamical matrix of NaCl in C and python implementations.

    1. Without NAC.
    2. Without NAC with comapact fc2.

    """
    if is_compact_fc:
        ph = ph_nacl_nonac_compact_fc
    else:
        ph = ph_nacl_nonac
    dynmat = ph.dynamical_matrix
    _test_dynmat(dynmat, lang=lang)
    _test_dynmat_252525(dynmat, dynmat_ref_252525, lang=lang)


def test_dynmat_without_hermitianize(ph_tipn3: Phonopy):
    """Test dynamical matrix of NaCl without hermitianization."""
    ph = ph_tipn3
    ph_no_H = Phonopy(
        unitcell=ph.unitcell,
        supercell_matrix=ph.supercell_matrix,
        primitive_matrix=ph.primitive_matrix,
        hermitianize_dynamical_matrix=False,
    )
    ph_no_H.force_constants = ph.force_constants
    ph_no_H.dynamical_matrix.run([0.1, 0.2, 0.3])
    dynmat = ph_no_H.dynamical_matrix.dynamical_matrix.copy().view("double").ravel()
    ph_no_H.nac_params = ph.nac_params
    ph_no_H.dynamical_matrix.run([0.1, 0.2, 0.3])
    dynmat_nac = ph_no_H.dynamical_matrix.dynamical_matrix.copy().view("double").ravel()

    # for v, v_nac in zip(
    #     dynmat, dynmat_nac
    # ):
    #     print(f"[{v:.6f}, {v_nac:.6f}],")

    dynmat_ref = [
        [0.246182, 0.307272],
        [0.000000, 0.000000],
        [-0.000000, -0.029119],
        [-0.000000, 0.000000],
        [-0.000000, 0.064382],
        [-0.009648, -0.009648],
        [-0.001231, 0.040164],
        [0.026915, 0.023644],
        [-0.047527, -0.092873],
        [0.024870, 0.022167],
        [0.078951, 0.162235],
        [0.077734, 0.077139],
        [-0.084303, -0.157220],
        [-0.001023, 0.014145],
        [0.009297, 0.097007],
        [-0.080021, -0.082182],
        [-0.023184, -0.098696],
        [-0.018592, -0.017667],
        [-0.107065, -0.141224],
        [-0.005219, -0.006211],
        [0.009112, 0.029925],
        [-0.010045, -0.008524],
        [-0.080001, -0.212950],
        [-0.001242, 0.002261],
        [-0.083587, -0.160020],
        [-0.007829, 0.002031],
        [0.015321, 0.021316],
        [0.080391, 0.076889],
        [-0.007518, -0.046121],
        [-0.019975, -0.020476],
        [-0.000000, -0.029119],
        [-0.000000, -0.000000],
        [0.515422, 0.518938],
        [-0.000000, 0.000000],
        [-0.000000, -0.019545],
        [-0.000000, -0.000000],
        [-0.018988, -0.033320],
        [0.009936, 0.009082],
        [-0.259304, -0.248647],
        [0.154188, 0.149243],
        [-0.015551, -0.042299],
        [-0.029717, -0.026716],
        [-0.002328, 0.022343],
        [-0.001161, -0.002281],
        [-0.235996, -0.273439],
        [0.131157, 0.113701],
        [0.032138, 0.058107],
        [-0.022431, -0.016702],
        [0.011156, 0.026933],
        [-0.012299, -0.011145],
        [0.022446, 0.018680],
        [-0.102197, -0.100718],
        [0.024527, 0.063315],
        [0.022247, 0.013741],
        [0.005636, 0.031334],
        [0.007996, 0.005632],
        [-0.293945, -0.279276],
        [-0.063835, -0.041604],
        [-0.050716, -0.033398],
        [-0.017157, -0.007532],
        [0.000000, 0.064382],
        [0.009648, 0.009648],
        [-0.000000, -0.019545],
        [0.000000, 0.000000],
        [0.482365, 0.501298],
        [-0.000000, -0.000000],
        [0.069733, 0.104311],
        [-0.012492, -0.012739],
        [-0.016992, -0.052129],
        [-0.032472, -0.028529],
        [-0.124116, -0.071818],
        [-0.104352, -0.073263],
        [-0.011267, -0.076237],
        [0.024675, 0.024813],
        [0.149603, 0.200770],
        [-0.080175, -0.057649],
        [-0.153260, -0.190855],
        [0.115963, 0.082074],
        [-0.029131, -0.067311],
        [0.020081, 0.021087],
        [0.005702, 0.020397],
        [0.005172, 0.001949],
        [-0.228644, -0.310719],
        [-0.294325, -0.287881],
        [-0.023853, -0.087870],
        [0.026147, 0.027485],
        [-0.120559, -0.103224],
        [0.054516, 0.027517],
        [-0.131398, -0.140758],
        [0.063685, 0.023709],
        [-0.001231, 0.040164],
        [-0.026915, -0.023644],
        [-0.018988, -0.033320],
        [-0.009936, -0.009082],
        [0.069733, 0.104311],
        [0.012492, 0.012739],
        [0.820024, 0.844377],
        [-0.000000, 0.000000],
        [0.000000, -0.036730],
        [0.000000, -0.000000],
        [-0.000000, 0.061821],
        [-0.066257, -0.066257],
        [-0.108498, -0.144853],
        [-0.017181, -0.025685],
        [0.045681, 0.104346],
        [-0.057560, -0.059583],
        [-0.040677, -0.089840],
        [0.053806, 0.055827],
        [-0.789005, -0.815957],
        [0.276959, 0.280367],
        [0.001859, 0.016101],
        [0.005062, 0.003218],
        [0.022220, -0.054501],
        [0.071625, 0.071497],
        [-0.120159, -0.160136],
        [0.005222, -0.000279],
        [-0.019361, -0.015161],
        [0.049779, 0.054722],
        [-0.024440, -0.048506],
        [0.052289, 0.055048],
        [-0.047527, -0.092873],
        [-0.024870, -0.022167],
        [-0.259304, -0.248647],
        [-0.154188, -0.149243],
        [-0.016992, -0.052129],
        [0.032472, 0.028529],
        [0.000000, -0.036730],
        [0.000000, 0.000000],
        [1.395941, 1.409971],
        [-0.000000, 0.000000],
        [-0.000000, -0.059381],
        [0.000000, 0.000000],
        [0.028257, 0.082800],
        [0.010674, 0.009359],
        [-0.505309, -0.583570],
        [-0.390639, -0.364103],
        [0.344338, 0.396667],
        [0.217636, 0.208176],
        [0.003265, 0.037423],
        [0.008889, 0.004467],
        [-0.198528, -0.204786],
        [0.096268, 0.093733],
        [0.062267, 0.135676],
        [-0.022870, -0.007908],
        [0.041317, 0.093110],
        [0.008775, 0.015770],
        [-0.693907, -0.660952],
        [0.003080, -0.030473],
        [-0.445378, -0.410492],
        [-0.040824, -0.054973],
        [0.078951, 0.162235],
        [-0.077734, -0.077139],
        [-0.015551, -0.042299],
        [0.029717, 0.026716],
        [-0.124116, -0.071818],
        [0.104352, 0.073263],
        [0.000000, 0.061821],
        [0.066257, 0.066257],
        [-0.000000, -0.059381],
        [-0.000000, -0.000000],
        [1.400913, 1.444700],
        [-0.000000, 0.000000],
        [-0.026670, -0.126276],
        [0.050024, 0.054687],
        [0.498375, 0.580447],
        [0.291571, 0.264758],
        [-0.487754, -0.552715],
        [-0.326309, -0.287700],
        [0.014924, -0.038146],
        [0.034324, 0.034235],
        [-0.010796, 0.010376],
        [0.003965, 0.008280],
        [-0.599843, -0.703997],
        [0.393903, 0.301895],
        [-0.048399, -0.145202],
        [0.032525, 0.035706],
        [-0.495586, -0.478477],
        [-0.163608, -0.131813],
        [-0.520624, -0.545131],
        [-0.171737, -0.125565],
        [-0.084303, -0.157220],
        [0.001023, -0.014145],
        [-0.002328, 0.022343],
        [0.001161, 0.002281],
        [-0.011267, -0.076237],
        [-0.024675, -0.024813],
        [-0.108498, -0.144853],
        [0.017181, 0.025685],
        [0.028257, 0.082800],
        [-0.010674, -0.009359],
        [-0.026670, -0.126276],
        [-0.050024, -0.054687],
        [0.567230, 0.638442],
        [-0.000000, 0.000000],
        [0.000000, -0.111381],
        [0.039170, 0.039170],
        [-0.000000, 0.091125],
        [0.030613, 0.030613],
        [-0.029852, 0.012217],
        [0.062189, 0.060366],
        [-0.046011, -0.066080],
        [-0.018681, -0.018502],
        [0.140941, 0.270872],
        [0.040334, 0.040953],
        [-0.052209, 0.017643],
        [-0.060516, -0.060378],
        [-0.042993, -0.047500],
        [0.074714, 0.075016],
        [-0.025216, 0.022677],
        [0.056896, 0.056984],
        [0.009297, 0.097007],
        [0.080021, 0.082182],
        [-0.235996, -0.273439],
        [-0.131157, -0.113701],
        [0.149603, 0.200770],
        [0.080175, 0.057649],
        [0.045681, 0.104346],
        [0.057560, 0.059583],
        [-0.505309, -0.583570],
        [0.390639, 0.364103],
        [0.498375, 0.580447],
        [-0.291571, -0.264758],
        [0.000000, -0.111381],
        [-0.039170, -0.039170],
        [2.510098, 2.587956],
        [0.000000, 0.000000],
        [-1.197668, -1.263892],
        [0.000000, -0.000000],
        [-0.068923, -0.121225],
        [-0.042748, -0.042612],
        [-0.144722, -0.109085],
        [0.109160, 0.112031],
        [0.166564, 0.040349],
        [-0.053277, -0.063747],
        [-0.042993, -0.148164],
        [0.074714, 0.075075],
        [-0.786906, -0.769344],
        [-0.022206, -0.023915],
        [-0.134648, -0.159735],
        [-0.025824, -0.026799],
        [-0.023184, -0.098696],
        [0.018592, 0.017667],
        [0.032138, 0.058107],
        [0.022431, 0.016702],
        [-0.153260, -0.190855],
        [-0.115963, -0.082074],
        [-0.040677, -0.089840],
        [-0.053806, -0.055827],
        [0.344338, 0.396667],
        [-0.217636, -0.208176],
        [-0.487754, -0.552715],
        [0.326309, 0.287700],
        [-0.000000, 0.091125],
        [-0.030613, -0.030613],
        [-1.197668, -1.263892],
        [0.000000, 0.000000],
        [1.112969, 1.161613],
        [0.000000, 0.000000],
        [0.104091, 0.148207],
        [0.067052, 0.067113],
        [0.209195, 0.186341],
        [-0.135690, -0.136797],
        [-0.158517, -0.053247],
        [0.046285, 0.060707],
        [0.025216, 0.115595],
        [-0.056896, -0.057066],
        [0.134648, 0.113586],
        [0.025824, 0.026275],
        [0.164443, 0.177332],
        [0.006806, 0.007153],
        [-0.107065, -0.141224],
        [0.005219, 0.006211],
        [0.011156, 0.026933],
        [0.012299, 0.011145],
        [-0.029131, -0.067311],
        [-0.020081, -0.021087],
        [-0.789005, -0.815957],
        [-0.276959, -0.280367],
        [0.003265, 0.037423],
        [-0.008889, -0.004467],
        [0.014924, -0.038146],
        [-0.034324, -0.034235],
        [-0.029852, 0.012217],
        [-0.062189, -0.060366],
        [-0.068923, -0.121225],
        [0.042748, 0.042612],
        [0.104091, 0.148207],
        [-0.067052, -0.067113],
        [1.729199, 1.753307],
        [0.000000, 0.000000],
        [0.000000, -0.015160],
        [0.000000, -0.000000],
        [-0.000000, 0.088472],
        [-0.055503, -0.055503],
        [-0.053697, -0.007943],
        [-0.033430, -0.032245],
        [0.022943, 0.018396],
        [-0.064929, -0.065496],
        [0.065763, 0.086691],
        [-0.083249, -0.083431],
        [0.009112, 0.029925],
        [0.010045, 0.008524],
        [0.022446, 0.018680],
        [0.102197, 0.100718],
        [0.005702, 0.020397],
        [-0.005172, -0.001949],
        [0.001859, 0.016101],
        [-0.005062, -0.003218],
        [-0.198528, -0.204786],
        [-0.096268, -0.093733],
        [-0.010796, 0.010376],
        [-0.003965, -0.008280],
        [-0.046011, -0.066080],
        [0.018681, 0.018502],
        [-0.144722, -0.109085],
        [-0.109160, -0.112031],
        [0.209195, 0.186341],
        [0.135690, 0.136797],
        [0.000000, -0.015160],
        [-0.000000, 0.000000],
        [0.581692, 0.584106],
        [0.000000, -0.000000],
        [0.000000, -0.035432],
        [0.000000, 0.000000],
        [0.012461, -0.007208],
        [-0.040541, -0.040974],
        [-0.108391, -0.125808],
        [-0.104691, -0.101061],
        [-0.242565, -0.258059],
        [-0.091879, -0.090393],
        [-0.080001, -0.212950],
        [0.001242, -0.002261],
        [0.024527, 0.063315],
        [-0.022247, -0.013741],
        [-0.228644, -0.310719],
        [0.294325, 0.287881],
        [0.022220, -0.054501],
        [-0.071625, -0.071497],
        [0.062267, 0.135676],
        [0.022870, 0.007908],
        [-0.599843, -0.703997],
        [-0.393903, -0.301895],
        [0.140941, 0.270872],
        [-0.040334, -0.040953],
        [0.166564, 0.040349],
        [0.053277, 0.063747],
        [-0.158517, -0.053247],
        [-0.046285, -0.060707],
        [0.000000, 0.088472],
        [0.055503, 0.055503],
        [0.000000, -0.035432],
        [-0.000000, -0.000000],
        [2.199008, 2.289599],
        [-0.000000, 0.000000],
        [0.127628, 0.251213],
        [-0.023512, -0.023308],
        [-0.301551, -0.311548],
        [-0.012477, -0.024988],
        [-0.233111, -0.181016],
        [0.003216, -0.014562],
        [-0.083587, -0.160020],
        [0.007829, -0.002031],
        [0.005636, 0.031334],
        [-0.007996, -0.005632],
        [-0.023853, -0.087870],
        [-0.026147, -0.027485],
        [-0.120159, -0.160136],
        [-0.005222, 0.000279],
        [0.041317, 0.093110],
        [-0.008775, -0.015770],
        [-0.048399, -0.145202],
        [-0.032525, -0.035706],
        [-0.052209, 0.017643],
        [0.060516, 0.060378],
        [-0.042993, -0.148164],
        [-0.074714, -0.075075],
        [0.025216, 0.115595],
        [0.056896, 0.057066],
        [-0.053697, -0.007943],
        [0.033430, 0.032245],
        [0.012461, -0.007208],
        [0.040541, 0.040974],
        [0.127628, 0.251213],
        [0.023512, 0.023308],
        [0.567230, 0.638442],
        [-0.000000, -0.000000],
        [-0.000000, -0.014082],
        [-0.039170, -0.039170],
        [0.000000, 0.042523],
        [0.030613, 0.030613],
        [0.015321, 0.021316],
        [-0.080391, -0.076889],
        [-0.293945, -0.279276],
        [0.063835, 0.041604],
        [-0.120559, -0.103224],
        [-0.054516, -0.027517],
        [-0.019361, -0.015161],
        [-0.049779, -0.054722],
        [-0.693907, -0.660952],
        [-0.003080, 0.030473],
        [-0.495586, -0.478477],
        [0.163608, 0.131813],
        [-0.042993, -0.047500],
        [-0.074714, -0.075016],
        [-0.786906, -0.769344],
        [0.022206, 0.023915],
        [0.134648, 0.113586],
        [-0.025824, -0.026275],
        [0.022943, 0.018396],
        [0.064929, 0.065496],
        [-0.108391, -0.125808],
        [0.104691, 0.101061],
        [-0.301551, -0.311548],
        [0.012477, 0.024988],
        [-0.000000, -0.014082],
        [0.039170, 0.039170],
        [2.510098, 2.478779],
        [-0.000000, -0.000000],
        [1.197668, 1.167764],
        [-0.000000, 0.000000],
        [-0.007518, -0.046121],
        [0.019975, 0.020476],
        [-0.050716, -0.033398],
        [0.017157, 0.007532],
        [-0.131398, -0.140758],
        [-0.063685, -0.023709],
        [-0.024440, -0.048506],
        [-0.052289, -0.055048],
        [-0.445378, -0.410492],
        [0.040824, 0.054973],
        [-0.520624, -0.545131],
        [0.171737, 0.125565],
        [-0.025216, 0.022677],
        [-0.056896, -0.056984],
        [-0.134648, -0.159735],
        [0.025824, 0.026799],
        [0.164443, 0.177332],
        [-0.006806, -0.007153],
        [0.065763, 0.086691],
        [0.083249, 0.083431],
        [-0.242565, -0.258059],
        [0.091879, 0.090393],
        [-0.233111, -0.181016],
        [-0.003216, 0.014562],
        [0.000000, 0.042523],
        [-0.030613, -0.030613],
        [1.197668, 1.167764],
        [-0.000000, -0.000000],
        [1.112969, 1.103520],
        [0.000000, -0.000000],
    ]

    np.testing.assert_allclose(np.vstack((dynmat, dynmat_nac)).T, dynmat_ref, atol=1e-5)


@pytest.mark.parametrize("lang", ["C", "Py"])
def test_dynmat_dense_svecs(ph_nacl_nonac_dense_svecs: Phonopy, lang: str):
    """Test with dense svecs."""
    ph = ph_nacl_nonac_dense_svecs
    dynmat = ph.dynamical_matrix
    _test_dynmat(dynmat, lang=lang)
    _test_dynmat_252525(dynmat, dynmat_ref_252525, lang=lang)


def test_dynmat_gonze_lee(ph_nacl: Phonopy):
    """Test with NAC by Gonze and Lee."""
    dynmat = ph_nacl.dynamical_matrix
    _test_dynmat_252525(dynmat, dynmat_gonze_lee_ref_252525)


def test_dynmat_gonze_lee_short_range_fc(ph_nacl: Phonopy):
    """Test force constants in dynamical matrix with NAC by Gonze and Lee."""
    # Test getter
    ph_nacl.dynamical_matrix.make_Gonze_nac_dataset()

    assert ph_nacl.dynamical_matrix._G_cutoff == pytest.approx(1.1584988384375283)
    assert ph_nacl.dynamical_matrix._G_list.shape == (307, 3)
    np.testing.assert_allclose(
        ph_nacl.dynamical_matrix._dd_q0.view("double").ravel(),
        [
            0.5509692730441111,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.5509692730441109,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.5509692730441113,
            0.0,
            0.5509692730441111,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.5509692730441109,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.5509692730441113,
            0.0,
        ],
        atol=1e-5,
    )
    fc = ph_nacl.dynamical_matrix.force_constants
    sr_fc = ph_nacl.dynamical_matrix.short_range_force_constants
    np.testing.assert_allclose(
        np.diag(fc[0, 1]), [-0.3017767, 0.0049673, 0.0049673], atol=1e-5
    )
    np.testing.assert_allclose(
        np.diag(sr_fc[0, 1]), [-0.13937495, -0.04645899, -0.04645899], atol=1e-5
    )

    # Test setter.
    ph_nacl.dynamical_matrix.short_range_force_constants = fc
    sr_fc = ph_nacl.dynamical_matrix.short_range_force_constants
    np.testing.assert_allclose(
        np.diag(sr_fc[0, 1]), [-0.3017767, 0.0049673, 0.0049673], atol=1e-5
    )
    ph_nacl.dynamical_matrix.make_Gonze_nac_dataset()
    sr_fc = ph_nacl.dynamical_matrix.short_range_force_constants
    np.testing.assert_allclose(
        np.diag(sr_fc[0, 1]), [-0.13937495, -0.04645899, -0.04645899], atol=1e-5
    )


def test_dynmat_gonze_lee_full_term(ph_nacl: Phonopy):
    """Test with NAC by Gonze and Lee."""
    dynmat = ph_nacl.dynamical_matrix
    _dynmat = DynamicalMatrixGL(
        dynmat.supercell,
        dynmat.primitive,
        dynmat.force_constants,
        nac_params=dynmat.nac_params,
        with_full_terms=True,
    )
    _test_dynmat_252525(_dynmat, dynmat_gonze_lee_full_ref_252525)


def test_dynmat_wang(ph_nacl_wang: Phonopy):
    """Test with NAC by Wang et al."""
    dynmat = ph_nacl_wang.dynamical_matrix
    _test_dynmat_252525(dynmat, dynmat_wang_ref_252525)


def _test_dynmat(dynmat: DynamicalMatrix, lang=None):
    dtype_complex = "c%d" % (np.dtype("double").itemsize * 2)
    if lang:
        dynmat.run([0, 0, 0], lang=lang)
    else:
        dynmat.run([0, 0, 0])
    dynmat_ref = (
        np.array(dynmat_ref_000, dtype="double").view(dtype=dtype_complex).reshape(6, 6)
    )
    np.testing.assert_allclose(dynmat.dynamical_matrix, dynmat_ref, atol=1e-5)

    if lang:
        dynmat.run([0.5, 0.5, 0.5], lang=lang)
    else:
        dynmat.run([0.5, 0.5, 0.5])
    dynmat_ref = (
        np.array(dynmat_ref_555, dtype="double").view(dtype=dtype_complex).reshape(6, 6)
    )
    np.testing.assert_allclose(dynmat.dynamical_matrix, dynmat_ref, atol=1e-5)


def _test_dynmat_252525(dynmat: DynamicalMatrix, dynmat_ref: list, lang=None):
    dtype_complex = "c%d" % (np.dtype("double").itemsize * 2)
    if lang:
        dynmat.run([0.25, 0.25, 0.25], lang=lang)
    else:
        dynmat.run([0.25, 0.25, 0.25])
    # for row in dynmat.dynamical_matrix:
    #     print("".join(["%f, %f, " % (c.real, c.imag) for c in row]))

    np.testing.assert_allclose(
        dynmat.dynamical_matrix,
        np.array(dynmat_ref, dtype="double").view(dtype=dtype_complex).reshape(6, 6),
        atol=1e-5,
    )
