//
// Context.h
//
// $Id: //poco/1.3/NetSSL_OpenSSL/include/Poco/Net/Context.h#7 $
//
// Library: NetSSL_OpenSSL
// Package: SSLCore
// Module:  Context
//
// Definition of the Context class.
//
// Copyright (c) 2006-2009, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef NetSSL_Context_INCLUDED
#define NetSSL_Context_INCLUDED


#include "Poco/Net/NetSSL.h"
#include "Poco/RefCountedObject.h"
#include "Poco/AutoPtr.h"
#include <openssl/ssl.h>


namespace Poco {
namespace Net {


class NetSSL_API Context: public Poco::RefCountedObject
	/// This class encapsulates context information for
	/// an SSL server or client, such as the certificate
	/// verification mode and the location of certificates
	/// and private key files, as well as the list of
	/// supported ciphers.
{
public:
	typedef Poco::AutoPtr<Context> Ptr;
	
	enum Usage
	{
		CLIENT_USE, /// Context is used by a client.
		SERVER_USE  /// Context is used by a server.
	};
	
	enum VerificationMode 
	{
		VERIFY_NONE    = SSL_VERIFY_NONE, 
			/// Server: The server will not send a client certificate 
			/// request to the client, so the client will not send a certificate. 
			///
			/// Client: If not using an anonymous cipher (by default disabled), 
			/// the server will send a certificate which will be checked, but
			/// the result of the check will be ignored.
			
		VERIFY_RELAXED = SSL_VERIFY_PEER, 
			/// Server: The server sends a client certificate request to the 
			/// client. The certificate returned (if any) is checked. 
			/// If the verification process fails, the TLS/SSL handshake is 
			/// immediately terminated with an alert message containing the 
			/// reason for the verification failure. 
			///
			/// Client: The server certificate is verified, if one is provided. 
			/// If the verification process fails, the TLS/SSL handshake is
			/// immediately terminated with an alert message containing the 
			/// reason for the verification failure. 
			
		VERIFY_STRICT  = SSL_VERIFY_PEER | SSL_VERIFY_FAIL_IF_NO_PEER_CERT,
			/// Server: If the client did not return a certificate, the TLS/SSL 
			/// handshake is immediately terminated with a handshake failure
			/// alert. 
			///
			/// Client: Same as VERIFY_RELAXED. 
			
		VERIFY_ONCE    = SSL_VERIFY_PEER | SSL_VERIFY_CLIENT_ONCE
			/// Server: Only request a client certificate on the initial 
			/// TLS/SSL handshake. Do not ask for a client certificate 
			/// again in case of a renegotiation.
			///
			/// Client: Same as VERIFY_RELAXED.	
	};

	Context(
		Usage usage,
		const std::string& privateKeyFile,
		const std::string& certificateFile,
		const std::string& caLocation, 
		VerificationMode verificationMode = VERIFY_RELAXED,
		int verificationDepth = 9,
		bool loadDefaultCAs = false,
		const std::string& cipherList = "ALL:!ADH:!LOW:!EXP:!MD5:@STRENGTH");
			/// Creates a Context.
			/// 
			///   * usage specifies whether the context is used by a client or server.
			///   * privateKeyFile contains the path to the private key file used for encryption.
			///     Can be empty if no private key file is used.
			///   * certificateFile contains the path to the certificate file (in PEM format).
			///     If the private key and the certificate are stored in the same file, this
			///     can be empty if privateKeyFile is given.
			///   * caLocation contains the path to the file or directory containing the
			///     CA/root certificates. Can be empty if the OpenSSL builtin CA certificates
			///     are used (see loadDefaultCAs).
			///   * verificationMode specifies whether and how peer certificates are validated.
			///   * verificationDepth sets the upper limit for verification chain sizes. Verification
			///     will fail if a certificate chain larger than this is encountered.
			///   * loadDefaultCAs specifies wheter the builtin CA certificates from OpenSSL are used.
			///   * cipherList specifies the supported ciphers in OpenSSL notation.

	~Context();
		/// Destroys the Context.

	SSL_CTX* sslContext() const;
		/// Returns the underlying OpenSSL SSL Context object.

	Usage usage() const;
		/// Returns whether the context is for use by a client or by a server.

	Context::VerificationMode verificationMode() const;
		/// Returns the verification mode.
		
	void enableSessionCache(bool flag = true);
		/// Enable or disable the SSL/TLS session cache for a server.
		///
		/// The default is a disabled session cache.
		
	bool sessionCacheEnabled() const;
		/// Returns true iff the session cache is enabled.

private:
	Usage _usage;
	VerificationMode _mode;
	SSL_CTX* _pSSLContext;
};


//
// inlines
//
inline Context::Usage Context::usage() const
{
	return _usage;
}


inline Context::VerificationMode Context::verificationMode() const
{
	return _mode;
}


inline SSL_CTX* Context::sslContext() const
{
	return _pSSLContext;
}


} } // namespace Poco::Net


#endif // NetSSL_Context_INCLUDED
