// Copyright (c) 2007, Google Inc.
// All rights reserved.
//
// Author: Li Liu
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
#include <cassert>
#include <signal.h>
#include <cstring>
#include <sys/types.h>
#include <sys/user.h>

#include "client/linux/handler/sigcontext.h"

namespace google_breakpad {

SignalContext::SignalContext() : sig_ctx_(NULL) {
}

void SignalContext::SetSignalContext(const struct sigcontext *sig_ctx) {
  assert(sig_ctx);
  sig_ctx_ = sig_ctx;
}

void SignalContext::SetUContext(const ucontext_t *ucontext) {
  assert(ucontext);
  // mcontex_t and sigcontext has the same definition.
  // We just do a convertion here.
  sig_ctx_ = reinterpret_cast<const sigcontext*>(&ucontext->uc_mcontext);
}

const struct sigcontext *SignalContext::GetRawContext() const {
  return sig_ctx_;
}

uintptr_t SignalContext::GetIP() const {
  assert(sig_ctx_);
#ifdef __i386__
  return sig_ctx_->eip;
#elif defined(__x86_64__)
  return sig_ctx_->rip;
#else
  assert(!"only work on i386 and x86_64!");
#endif
}

uintptr_t SignalContext::GetFramePointer() const {
  assert(sig_ctx_);
#ifdef __i386__
  return sig_ctx_->ebp;
#elif defined(__x86_64__)
  return sig_ctx_->rbp;
#else
  assert(!"only work on i386 and x86_64!");
#endif
}

uintptr_t SignalContext::GetStackPointer() const {
  assert(sig_ctx_);
#ifdef __i386__
  return sig_ctx_->esp;
#elif defined(__x86_64__)
  return sig_ctx_->rsp;
#else
  assert(!"only work on i386 and x86_64!");
#endif
}

bool SignalContext::IsEmpty() const {
  return sig_ctx_ == NULL;
}

#ifdef __i386__
bool SignalContext::CopyTo(RawContext *context) const {
  context->context_flags = MD_CONTEXT_X86_FULL;
  context->gs = sig_ctx_->gs;
  context->fs = sig_ctx_->fs;
  context->es = sig_ctx_->es;
  context->ds = sig_ctx_->ds;
  context->cs = sig_ctx_->cs;
  context->ss = sig_ctx_->ss;
  context->edi = sig_ctx_->edi;
  context->esi = sig_ctx_->esi;
  context->ebp = sig_ctx_->ebp;
  context->esp = sig_ctx_->esp;
  context->ebx = sig_ctx_->ebx;
  context->edx = sig_ctx_->edx;
  context->ecx = sig_ctx_->ecx;
  context->eax = sig_ctx_->eax;
  context->eip = sig_ctx_->eip;
  context->eflags = sig_ctx_->eflags;
  if (sig_ctx_->fpstate != NULL) {
    context->context_flags = MD_CONTEXT_X86_FULL |
      MD_CONTEXT_X86_FLOATING_POINT;
    context->float_save.control_word = sig_ctx_->fpstate->cw;
    context->float_save.status_word = sig_ctx_->fpstate->sw;
    context->float_save.tag_word = sig_ctx_->fpstate->tag;
    context->float_save.error_offset = sig_ctx_->fpstate->ipoff;
    context->float_save.error_selector = sig_ctx_->fpstate->cssel;
    context->float_save.data_offset = sig_ctx_->fpstate->dataoff;
    context->float_save.data_selector = sig_ctx_->fpstate->datasel;
    memcpy(context->float_save.register_area, sig_ctx_->fpstate->_st,
           sizeof(context->float_save.register_area));
  }
  return true;
}
#endif

#ifdef __x86_64__
bool SignalContext::CopyTo(RawContext *context) const {
  context->context_flags = MD_CONTEXT_AMD64_CONTROL |
                           MD_CONTEXT_AMD64_INTEGER |
                           MD_CONTEXT_AMD64_SEGMENTS;
  context->cs = sig_ctx_->cs;
    context->fs = sig_ctx_->fs;
  context->gs = sig_ctx_->gs;
  context->eflags = sig_ctx_->eflags;
  context->rip = sig_ctx_->rip;
  context->rax = sig_ctx_->rax;
  context->rbx = sig_ctx_->rbx;
  context->rcx = sig_ctx_->rcx;
  context->rdx = sig_ctx_->rdx;
  context->rsp = sig_ctx_->rsp;
  context->rbp = sig_ctx_->rbp;
  context->rsi = sig_ctx_->rsi;
  context->rdi = sig_ctx_->rdi;
  context->r8 = sig_ctx_->r8;
  context->r9 = sig_ctx_->r9;
  context->r10 = sig_ctx_->r10;
  context->r11 = sig_ctx_->r11;
  context->r12 = sig_ctx_->r12;
  context->r13 = sig_ctx_->r13;
  context->r14 = sig_ctx_->r14;
  context->r15 = sig_ctx_->r15;

  // The following registers are not set by kernel
  // when setting up signal handler frame.
  context->ds = 0;
  context->es = 0;
  context->ss = 0;
  return true;
}
#endif

}  // namespace google_breakpad
