/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "common/file.h"

#include "freescape/freescape.h"
#include "freescape/games/castle/castle.h"
#include "freescape/language/8bitDetokeniser.h"

namespace Freescape {

void CastleEngine::initZX() {
	_viewArea = Common::Rect(64, 36, 256, 148);
	_soundIndexShoot = 5;
	_soundIndexCollide = 3;
	_soundIndexStartFalling = -1;
	_soundIndexFallen = 1;
	_soundIndexFall = 6;
	_soundIndexStepUp = 12;
	_soundIndexStepDown = 12;
	_soundIndexMenu = 3;
	_soundIndexStart = 7;
	_soundIndexAreaChange = 7;
}

Graphics::ManagedSurface *CastleEngine::loadFrameWithHeader(Common::SeekableReadStream *file, int pos, uint32 front, uint32 back) {
	Graphics::ManagedSurface *surface = new Graphics::ManagedSurface();
	file->seek(pos);
	int16 width = file->readByte();
	int16 height = file->readByte();
	debugC(kFreescapeDebugParser, "Frame size: %d x %d", width, height);
	surface->create(width * 8, height, _gfx->_texturePixelFormat);

	/*byte mask =*/ file->readByte();

	surface->fillRect(Common::Rect(0, 0, width * 8, height), back);
	/*int frameSize =*/ file->readUint16LE();
	return loadFrame(file, surface, width, height, front);
}

Common::Array<Graphics::ManagedSurface *> CastleEngine::loadFramesWithHeader(Common::SeekableReadStream *file, int pos, int numFrames, uint32 front, uint32 back) {
	Graphics::ManagedSurface *surface = nullptr;
	file->seek(pos);
	int16 width = file->readByte();
	int16 height = file->readByte();
	/*byte mask =*/ file->readByte();

	/*int frameSize =*/ file->readUint16LE();
	Common::Array<Graphics::ManagedSurface *> frames;
	for (int i = 0; i < numFrames; i++) {
		surface = new Graphics::ManagedSurface();
		surface->create(width * 8, height, _gfx->_texturePixelFormat);
		surface->fillRect(Common::Rect(0, 0, width * 8, height), back);
		frames.push_back(loadFrame(file, surface, width, height, front));
	}

	return frames;
}


Graphics::ManagedSurface *CastleEngine::loadFrame(Common::SeekableReadStream *file, Graphics::ManagedSurface *surface, int width, int height, uint32 front) {
	for (int i = 0; i < width * height; i++) {
		byte color = file->readByte();
		for (int n = 0; n < 8; n++) {
			int y = i / width;
			int x = (i % width) * 8 + (7 - n);
			if ((color & (1 << n)))
				surface->setPixel(x, y, front);
		}
	}
	return surface;
}

void CastleEngine::loadAssetsZXFullGame() {
	Common::File file;
	uint8 r, g, b;
	Common::Array<Graphics::ManagedSurface *> chars;

	file.open("castlemaster.zx.title");
	if (file.isOpen()) {
		_title = loadAndConvertScrImage(&file);
	} else
		error("Unable to find castlemaster.zx.title");

	file.close();
	file.open("castlemaster.zx.border");
	if (file.isOpen()) {
		_border = loadAndConvertScrImage(&file);
	} else
		error("Unable to find castlemaster.zx.border");
	file.close();

	file.open("castlemaster.zx.data");
	if (!file.isOpen())
		error("Failed to open castlemaster.zx.data");

	loadMessagesVariableSize(&file, 0x4bd, 71);
	switch (_language) {
		case Common::ES_ESP:
			loadRiddles(&file, 0x1458, 9);
			load8bitBinary(&file, 0x6aa9, 16);
			loadSpeakerFxZX(&file, 0xca0, 0xcdc);

			file.seek(0x1228);
			for (int i = 0; i < 90; i++) {
				Graphics::ManagedSurface *surface = new Graphics::ManagedSurface();
				surface->create(8, 8, Graphics::PixelFormat::createFormatCLUT8());
				chars.push_back(loadFrame(&file, surface, 1, 8, 1));
			}
			_font = Font(chars);
			_font.setCharWidth(9);
			_fontLoaded = true;

			break;
		case Common::EN_ANY:
			if (_variant & GF_ZX_RETAIL) {
				loadRiddles(&file, 0x1448, 9);
				load8bitBinary(&file, 0x6a3b, 16);
				loadSpeakerFxZX(&file, 0xc91, 0xccd);
				file.seek(0x1219);
			} else if (_variant & GF_ZX_DISC) {
				loadRiddles(&file, 0x1457, 9);
				load8bitBinary(&file, 0x6a9b, 16);
				loadSpeakerFxZX(&file, 0xca0, 0xcdc);
				file.seek(0x1228);
			} else {
				error("Unknown Castle Master ZX variant");
			}
			for (int i = 0; i < 90; i++) {
				Graphics::ManagedSurface *surface = new Graphics::ManagedSurface();
				surface->create(8, 8, Graphics::PixelFormat::createFormatCLUT8());
				chars.push_back(loadFrame(&file, surface, 1, 8, 1));
			}
			_font = Font(chars);
			_font.setCharWidth(9);
			_fontLoaded = true;

			break;
		default:
			error("Language not supported");
			break;
	}

	loadColorPalette();
	_gfx->readFromPalette(2, r, g, b);
	uint32 red = _gfx->_texturePixelFormat.ARGBToColor(0xFF, r, g, b);

	_gfx->readFromPalette(7, r, g, b);
	uint32 white = _gfx->_texturePixelFormat.ARGBToColor(0xFF, r, g, b);

	_keysBorderFrames.push_back(loadFrameWithHeader(&file, _variant & GF_ZX_DISC ? 0xe06 : 0xdf7, red, white));

	uint32 green = _gfx->_texturePixelFormat.ARGBToColor(0xFF, 0, 0xff, 0);
	_spiritsMeterIndicatorFrame = loadFrameWithHeader(&file, _variant & GF_ZX_DISC ? 0xe5e : 0xe4f, green, white);

	_gfx->readFromPalette(4, r, g, b);
	uint32 front = _gfx->_texturePixelFormat.ARGBToColor(0xFF, r, g, b);

	int backgroundWidth = 16;
	int backgroundHeight = 18;
	Graphics::ManagedSurface *background = new Graphics::ManagedSurface();
	background->create(backgroundWidth * 8, backgroundHeight, _gfx->_texturePixelFormat);
	background->fillRect(Common::Rect(0, 0, backgroundWidth * 8, backgroundHeight), 0);

	file.seek(_variant & GF_ZX_DISC ? 0xfd3 : 0xfc4);
	_background = loadFrame(&file, background, backgroundWidth, backgroundHeight, front);

	_gfx->readFromPalette(6, r, g, b);
	uint32 yellow = _gfx->_texturePixelFormat.ARGBToColor(0xFF, r, g, b);
	uint32 black = _gfx->_texturePixelFormat.ARGBToColor(0xFF, 0, 0, 0);
	_strenghtBackgroundFrame = loadFrameWithHeader(&file, _variant & GF_ZX_DISC ? 0xee6 : 0xed7, yellow, black);
	_strenghtBarFrame = loadFrameWithHeader(&file, _variant & GF_ZX_DISC ? 0xf72 : 0xf63, yellow, black);
	_strenghtWeightsFrames = loadFramesWithHeader(&file, _variant & GF_ZX_DISC ? 0xf92 : 0xf83, 4, yellow, black);

	_flagFrames = loadFramesWithHeader(&file, (_variant & GF_ZX_DISC ? 0x10e4 + 15 : 0x10e4), 4, green, black);

	file.skip(24);
	int thunderWidth = 4;
	int thunderHeight = 44;
	Graphics::ManagedSurface *thunderFrame = new Graphics::ManagedSurface();
	thunderFrame->create(thunderWidth * 8, thunderHeight, _gfx->_texturePixelFormat);
	thunderFrame->fillRect(Common::Rect(0, 0, thunderWidth * 8, thunderHeight), 0);
	thunderFrame = loadFrame(&file, thunderFrame, thunderWidth, thunderHeight, front);

	_thunderFrames.push_back(new Graphics::ManagedSurface);
	_thunderFrames.push_back(new Graphics::ManagedSurface);

	_thunderFrames[0]->create(thunderWidth * 8 / 2, thunderHeight, _gfx->_texturePixelFormat);
	_thunderFrames[1]->create(thunderWidth * 8 / 2, thunderHeight, _gfx->_texturePixelFormat);

	_thunderFrames[0]->copyRectToSurface(*thunderFrame, 0, 0, Common::Rect(0, 0, thunderWidth * 8 / 2, thunderHeight));
	_thunderFrames[1]->copyRectToSurface(*thunderFrame, 0, 0, Common::Rect(thunderWidth * 8 / 2, 0, thunderWidth * 8, thunderHeight));

	Graphics::Surface *tmp;
	tmp = loadBundledImage("castle_riddle_top_frame");
	_riddleTopFrame = new Graphics::ManagedSurface;
	_riddleTopFrame->copyFrom(*tmp);
	tmp->free();
	delete tmp;
	_riddleTopFrame->convertToInPlace(_gfx->_texturePixelFormat);

	tmp = loadBundledImage("castle_riddle_background_frame");
	_riddleBackgroundFrame = new Graphics::ManagedSurface();
	_riddleBackgroundFrame->copyFrom(*tmp);
	tmp->free();
	delete tmp;
	_riddleBackgroundFrame->convertToInPlace(_gfx->_texturePixelFormat);

	tmp = loadBundledImage("castle_riddle_bottom_frame");
	_riddleBottomFrame = new Graphics::ManagedSurface();
	_riddleBottomFrame->copyFrom(*tmp);
	tmp->free();
	delete tmp;
	_riddleBottomFrame->convertToInPlace(_gfx->_texturePixelFormat);
}

void CastleEngine::drawZXUI(Graphics::Surface *surface) {
	uint32 color = 5;
	uint8 r, g, b;

	drawLiftingGate(surface);
	drawDroppingGate(surface);

	_gfx->readFromPalette(color, r, g, b);
	uint32 front = _gfx->_texturePixelFormat.ARGBToColor(0xFF, r, g, b);

	color = 0;
	_gfx->readFromPalette(color, r, g, b);
	uint32 black = _gfx->_texturePixelFormat.ARGBToColor(0xFF, r, g, b);

	Common::Rect backRect(123, 179, 242 + 5, 188);
	surface->fillRect(backRect, black);

	Common::String message;
	int deadline = -1;
	getLatestMessages(message, deadline);
	if (deadline > 0 && deadline <= _countdown) {
		//debug("deadline: %d countdown: %d", deadline, _countdown);
		drawStringInSurface(message, 120, 179, front, black, surface);
		_temporaryMessages.push_back(message);
		_temporaryMessageDeadlines.push_back(deadline);
	} else {
		if (_gameStateControl != kFreescapeGameStateEnd) {
			if (getGameBit(31)) { // The final cutscene is playing but it is not ended yet
				drawStringInSurface(_messagesList[5], 120, 179, front, black, surface); // "You did it!"
			} else
				drawStringInSurface(_currentArea->_name, 120, 179, front, black, surface);
		}
	}

	for (int k = 0; k < int(_keysCollected.size()); k++) {
		surface->copyRectToSurface((const Graphics::Surface)*_keysBorderFrames[0], 99 - k * 4, 177, Common::Rect(0, 0, 6, 11));
	}

	uint32 green = _gfx->_texturePixelFormat.ARGBToColor(0xFF, 0, 0xff, 0);

	surface->fillRect(Common::Rect(152, 156, 216, 164), green);
	surface->copyRectToSurface((const Graphics::Surface)*_spiritsMeterIndicatorFrame, 140 + _spiritsMeterPosition, 156, Common::Rect(0, 0, 15, 8));

	surface->fillRect(Common::Rect(64, 155, 64 + 72, 155 + 15), _gfx->_texturePixelFormat.ARGBToColor(0xFF, 0x00, 0x00, 0x00));
	drawEnergyMeter(surface, Common::Point(64, 155));

	int ticks = g_system->getMillis() / 20;
	int flagFrameIndex = (ticks / 10) % 4;
	surface->copyRectToSurface(*_flagFrames[flagFrameIndex], 264, 9, Common::Rect(0, 0, _flagFrames[flagFrameIndex]->w, _flagFrames[flagFrameIndex]->h));
}

} // End of namespace Freescape
