use ui;
use graphics;
use core:geometry;
use progvis:data;

/**
 * A workspace, where we place all variables and other data structures currently
 * visible to the user.
 */
class Workspace on Render {
	init() {
		init() {
			outputWidth = 300;
			enableStepFn = &alwaysTrue;
		}
	}

	// Override default width of threads? (if positive)
	Size threadSize;

	// Override the default width of the output panel?
	Float outputWidth;

	// Check if stepping of threads should be enabled.
	private Fn<Bool> enableStepFn;

	// Set stepping.
	assign enableStep(Fn<Bool> v) {
		enableStepFn = v;
		for (e in elements)
			if (x = e.visual.contained as ThreadView)
				x.enableStepFn = v;
	}

	// Clear all marks.
	void clearNext() {
		for (e in elements)
			if (x = e.visual.contained as ThreadView)
				x.clearNext();
	}

	// Mark a particular thread as being the next step.
	void markNext(Nat thread, Bool barrier) {
		for (e in elements)
			if (x = e.visual.contained as ThreadView)
				if (x.threadId == thread)
					x.markNext(barrier);
	}

	// Draw the workspace. Assumes we can clip anything not inside 'viewport'.
	void render(Rect viewport, Graphics g) {
		// Find new docked elements.
		for (e in elements) {
			if (!e.docked) {
				if (e.visual.docked) {
					e.docked = true;
					docked.push(e);
				}
			}
		}

		layoutDocked(viewport);

		for (e in elements) {
			// TODO: Clipping?
			if (!e.topmost) {
				var visual = e.visual;
				visual.draw(g, active is visual);
			}
		}

		// Draw elements that are topmost.
		for (i, e in elements) {
			if (e.topmost) {
				var visual = e.visual;
				visual.draw(g, active is visual);
			}
		}

		// Draw pointers etc above all other elements.
		for (i, e in elements) {
			var visual = e.visual;
			visual.drawLinks(g, active is visual);
		}
	}

	// Layout docked elements.
	private void layoutDocked(Rect viewport) {
		Float space = layoutEdgeSpace;
		Float x = space + viewport.p0.x;

		for (Nat i = 0; i < docked.count; i++) {
			var e = docked[i];
			// Note: e.docked is set to false when the element is removed.
			if (!e.visual.docked | !e.docked) {
				e.docked = false;
				docked.remove(i);
				i--;
				continue;
			}

			if (e.thread) {
				Size sz = e.visual.size;
				e.visual.pos.x = x;
				e.visual.pos.y = viewport.p1.y - space - sz.h;

				x += sz.w + space;
			}
			if (e.output) {
				Size sz = e.visual.size;
				e.visual.pos.x = viewport.p1.x - sz.w - space;
				// e.visual.pos.y = viewport.p1.y - sz.h - space;
				e.visual.pos.y = viewport.p0.y + space;
			}
		}
	}

	// Helper for 'findAt', only considers either topmost or non-topmost elements.
	private Drawable? findAt(Point pt, Bool activate, Bool topmost) {
		for (Nat i = elements.count; i > 0; i--) {
			var elem = elements[i-1];
			if (elem.topmost != topmost)
				continue;

			Drawable d = elem.visual;
			if (d.rect.contains(pt)) {
				if (activate) {
					active = d;

					// Move it last to make it get focus.
					elements.remove(i-1);
					elements.push(elem);
				}
				return d.findAt(pt);
			}
		}

		return null;
	}

	// Find an object at the specified location.
	private Drawable? findAt(Point pt, Bool activate) {
		if (r = findAt(pt, activate, true))
			return r;
		if (r = findAt(pt, activate, false))
			return r;

		active = null;
		return null;
	}
	Drawable? findAt(Point pt) { findAt(pt, false); }
	Drawable? findAtActivate(Point pt) { findAt(pt, true); }
	void deactivate() { active = null; }

	/**
	 * Output.
	 */

	// Our output object, if we created it.
	private OutputText? output;

	// Get the output, create it if it does not exist already.
	private OutputText getOutput() {
		if (output)
			return output;

		OutputText created(outputWidth);
		Element elem(null, Dockable(created, "Output"));
		elem.output = true;
		elem.topmost = true;
		elements << elem;
		output = created;
		created;
	}

	void outputText(Str text) {
		var out = getOutput();
		out.add(text);
	}

	// Remove everything in the workspace.
	void clear() {
		elements.clear();
		docked.clear();
		output = null;
	}

	// Update the contents of the workspace from a world, and a program (for output).
	void update(World world) {
		Update update(this, world);

		Allocation->Nat allocToElem;
		for (id, elem in elements) {
			if (alloc = elem.alloc)
				allocToElem.put(alloc, id);
			elem.visited = false;
		}

		for (alloc in world) {
			Nat id = allocToElem.get(alloc, elements.count);
			if (id >= elements.count) {
				var elem = update.create(alloc);
				elements << elem;
				allocToElem.put(alloc, id);

				update.update(elem);
			} else {
				update.update(elements[id]);
			}
		}

		// Retry the update phase for elements that did not find pointer targets.
		update.retryUpdates();

		// Check elements that need layout.
		update.updateLayout();

		// Remove stale elements.
		Nat to = 0;
		for (Nat i = 0; i < elements.count; i++) {
			Element here = elements[i];

			if (to < i)
				elements[to] = here;
			if (here.visited)
				to++;
			else if (!here.visual.remove)
				to++;
			else
				here.docked = false;
		}
		while (to < elements.count) {
			elements.pop();
		}
	}

	// Resize thread if needed.
	void resizeThread(ThreadView thread) {
		unless (min(threadSize) > 1.0)
			return;

		thread.resize(threadSize);
	}

	/**
	 * State kept only while updating the view.
	 */
	class Update on Render {
		init(Workspace owner, World world) {
			init {
				world = world;
				owner = owner;
			}
		}

		// World.
		World world;

		// Owner.
		Workspace owner;

		// Map of our representation of various data elements.
		private Data->Drawable dataVisual;

		// Remember what the root element for each element. We build this during the update so that
		// we can use it for layout if needed.
		private Drawable->Element root;

		// Remember drawables that we did not find a data element for.
		private Drawable[] retryUpdate;

		// Elements that need to have updated layout.
		private Element[] needLayout;

		// Current element we are updating.
		private Element? updating;

		// Call to update an element. Sets up the class to keep track of which element is currently
		// being updated.
		void update(Element elem) {
			updating = elem;

			elem.visited = true;
			elem.visual.update(this);

			updating = null;
		}
		void update(Drawable elem) {
			var found = root.find(elem);
			if (found != root.end)
				updating = found.v;

			elem.update(this);

			updating = null;
		}

		// Create an element for an allocation.
		Element create(Allocation alloc) {
			var created = create(alloc.data);

			// Ask for modifications.
			for (h in world.hints) {
				created = h.decorateAllocation(alloc, created);
			}

			// Set stepping enabled for any new threads.
			if (x = created.contained as ThreadView)
				x.enableStepFn = owner.enableStepFn;

			var elem = if (alloc.data as progvis:data:Thread) {
				Element e(alloc, Dockable(created, alloc.data.title(world)).allowResize);
				e.thread = true;
				e.topmost = true;
				e;
			} else {
				// Don't have double borders.
				if (x = created as WithCaption)
					created = x.wrap;
				Element(alloc, Movable(created, alloc.data.title(world)));
			};

			needLayout << elem;
			elem;
		}

		// Create an element for a data element.
		Drawable create(Data data) {
			for (h in world.hints) {
				if (c = h.createView(world, data)) {
					if (c as ThreadView)
						owner.resizeThread(c);
					return c;
				}
			}

			return UnknownView(data);
		}

		// Find our representation of some data element. If a suitable element is not found, the
		// 'update' method of 'from' will be called again when we have traversed more elements.
		Drawable? find(Drawable from, Data find) {
			var x = dataVisual.find(find);
			if (x != dataVisual.end)
				return x.v;

			retryUpdate << from;
			return null;
		}

		// Register a representation of some data. Expected to be called during 'update' of elements.
		void setVisual(Data data, Drawable visual) {
			dataVisual.put(data, visual);
			if (updating)
				root.put(visual, updating);
		}

		// Retry 'find' operations that failed.
		void retryUpdates () {
			Nat to = retryUpdate.count; // in case of multiple failures.
			for (Nat i = 0; i < to; i++) {
				update(retryUpdate[i]);
			}

			if (to != retryUpdate.count)
				print("WARNING: Some pointers failed to find their targets.");
			retryUpdate.clear();
		}

		// Update any objects that need to be laid out properly.
		void updateLayout() {
			// Skip building the pointer map if we don't need it.
			if (needLayout.empty)
				return;

			// Build a representation of all pointers in the system. This is a high-level
			// representation that is more or less only suitable for doing layout (in most cases we
			// want to point to a particular part of an element, here we are only concerned with the
			// top-level elements). This maps stores what elements (value) point to a particular
			// element (key).
			Element->Set<Element> pointersTo;

			for (v in dataVisual) {
				unless (v as PointerView)
					continue;
				unless (to = v.target)
					continue;

				var srcParent = root.find(v);
				var dstParent = root.find(to);
				if (srcParent == root.end | dstParent == root.end)
					continue;

				// We don't add docked objects here as we are not interested in references from them
				// (it would make it very likely that a new thread would be placed on top of data).
				if (srcParent.v.docked)
					continue;
				if (srcParent.v.visual.docked)
					continue;

				pointersTo[dstParent.v].put(srcParent.v);
			}

			// Create a set of rectangles for quicker layout. Don't add docked objects or objects
			// that have not been laid out yet.
			Rect bounds;
			for (x in owner.elements)
				bounds = bounds.include(x.visual.rect);

			RectSet rects(bounds);
			for (x in owner.elements) {
				if (!x.docked & !x.needLayout)
					rects.add(x.visual.rect);
			}

			for (x in needLayout) {
				if (!updateLayout(x, pointersTo, rects)) {
					// This is a fallback in case of isolated cycles (which should not happen).
					placeElement(rects, x, Point(layoutEdgeSpace, layoutEdgeSpace));
				}
			}
		}

		// Update layout for a particular element if necessary. Returns 'true' if we managed to find
		// a layout, or 'false' if none of our parents had a position and we need to try again later.
		private Bool updateLayout(Element elem, Element->Set<Element> pointers, RectSet rects) {
			if (elem.topmost)
				elem.needLayout = false;
			if (!elem.needLayout)
				return true;

			// If we recursively find this element, exit. That means we have found a cycle.
			if (elem.doingLayout)
				return false;

			// Prevent cycles. We don't need to do this earlier since we do not have any recursive
			// calls to updateLayout before here.
			elem.doingLayout = true;

			// We try to place objects in columns of equal depth. This makes the pointer
			// visualization look pretty nice. So, we essentially only have to keep track of the
			// top-right corner of any parents. If multiple parents exist, the x coordinate is the
			// maximum of the parents, and the y coordinate is the minimum.
			var p = pointers[elem];
			Float xSpace = layoutObjSpace.w;
			Point suggested(layoutEdgeSpace, layoutEdgeSpace);
			Bool anyParent = false;
			for (src in p) {
				// If the element does not yet have a layout, try to give it a position.
				if (src.needLayout)
					if (!updateLayout(elem, pointers, rects))
						continue;

				// Update suggested as noted above.
				suggested.x = max(suggested.x, src.visual.rect.p1.x + xSpace);
				if (anyParent)
					suggested.y = min(suggested.y, src.visual.pos.y);
				else
					suggested.y = src.visual.pos.y;

				anyParent = true;
			}

			elem.doingLayout = false;

			// If we did not find any parent, we are in one of two cases. Either, we don't have any
			// parents (e.g. the Globals object, or something only referred to from docked objects),
			// or all calls to updateLayout failed. The first case is OK, while the second should
			// result in failure.
			if (!anyParent & p.any)
				return false;

			elem.needLayout = false;

			// Now we need where we want the element. Check where other elements are and place it
			// appropriately.
			placeElement(rects, elem, suggested);
			true;
		}

		// Place an element as close to the suggestion as possible, but picks a location where it
		// does not intersect with other elements.
		private void placeElement(RectSet rects, Element elem, Point suggestion) {
			Size pad(layoutObjSpace);

			Rect rect = elem.visual.rect.at(suggestion);
			// Grow it to get suitable padding.
			rect = rect.grow(pad);
			Point location = rects.fit(rect);

			elem.visual.pos = location + pad;
			rects.add(elem.visual.rect);
		}

	}

private:
	/**
	 * A single element on the screen with some additional metadata.
	 */
	class Element on Render {
		// Which allocation is this element associated with?
		Allocation? alloc;

		// Visual part of the object.
		Drawable visual;

		// Is this a thread? (impacts layout etc.)
		Bool thread;

		// Is this an output panel? (Put on the top right).
		Bool output;

		// Always on top?
		Bool topmost;

		// Is the element docked? (i.e. is it in the "docked"-array?)
		Bool docked;

		// Visited this round?
		Bool visited;

		// Need layout?
		Bool needLayout;

		// Are we doing layout for this element right now?
		Bool doingLayout;

		// Create.
		init(Allocation? alloc, Drawable visual) {
			init {
				alloc = alloc;
				visual = visual;
				thread = false;
				topmost = false;
				visited = true;
				needLayout = true;
			}
		}
	}

	// All elements in the workspace, in drawing order (back to front).
	Element[] elements;

	// Docked elements, to keep track of the order they were docked in.
	Element[] docked;

	// Active drawable.
	Drawable? active;
}

package Bool alwaysTrue() {
	true;
}
