/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#ifndef INDII_CLUSTER_DATASET_HPP
#define INDII_CLUSTER_DATASET_HPP

#include "ClusterVector.hpp"

#include <map>

namespace indii {
/**
 * Weighted data set for clustering.
 *
 * @tparam T Datum type.
 * @tparam W Weight type.
 */
template <class T, class W>
class DataSet {
public:
  /**
   * Vector comparison for map.
   */
  struct vectorCmp {
    bool operator()(const typename ClusterVector<T>::type& x1,
        const typename ClusterVector<T>::type& x2) const {
      typename ClusterVector<T>::type::const_iterator iter1, iter2, end1;

      iter1 = x1.begin();
      iter2 = x2.begin();
      end1 = x1.end();
      while (iter1 != end1) {
        if (*iter1 == *iter2) {
          iter1++;
          iter2++;
        } else {
          return *iter1 < *iter2;
        }
      }
      return false;
    }
  };

  /**
   * Data set.
   */
  typedef std::map<typename ClusterVector<T>::type,W,vectorCmp> data_set;

  /**
   * Data set iterator.
   */
  typedef typename data_set::iterator data_set_iterator;

  /**
   * Data set constant iterator.
   */
  typedef typename data_set::const_iterator data_set_const_iterator;

  /**
   * Get dimensionality of data set.
   *
   * @return Dimensionality of data set.
   */
  static unsigned int getDimensions();

  /**
   * Get number of points in data set.
   *
   * @return Number of points in data set.
   */
  unsigned int getSize() const;

  /**
   * Add datum.
   *
   * @param x The datum.
   * @param w Unnormalized weight of the datum.
   */
  void add(const typename ClusterVector<T>::type& x, const W w = 1);

  /**
   * Clear the data set.
   */
  void clear();

  /**
   * Get iterator to the beginning of the data set.
   *
   * @return Iterator to the beginning of the data set.
   */
  data_set_const_iterator begin() const;

  /**
   * Get iterator to the end of the data set.
   *
   * @return Iterator to the end of the data set.
   */
  data_set_const_iterator end() const;

private:
  /**
   * Data set.
   */
  data_set data;

};
}

template <class T, class W>
unsigned int indii::DataSet<T,W>::getDimensions() {
  return ClusterVector<T>::N;
}

template <class T, class W>
unsigned int indii::DataSet<T,W>::getSize() const {
  return (unsigned int)data.size();
}

template <class T, class W>
void indii::DataSet<T,W>::add(
    const typename ClusterVector<T>::type& x, const W w) {
  /* pre-condition */
  assert (x.size() == ClusterVector<T>::N);

  data_set_iterator find = data.find(x);
  if (find != data.end()) {
    find->second += w;
  } else {
    data.insert(make_pair(x, w));
  }
}

template <class T, class W>
void indii::DataSet<T,W>::clear() {
  data.clear();
}

template <class T, class W>
typename indii::DataSet<T,W>::data_set_const_iterator
    indii::DataSet<T,W>::begin() const {
  return data.begin();
}

template <class T, class W>
typename indii::DataSet<T,W>::data_set_const_iterator
    indii::DataSet<T,W>::end() const {
  return data.end();
}

#endif
