/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/
#include "shortcutui.h"

ShortcutUi::ShortcutUi()
{
    setAttribute(Qt::WA_DeleteOnClose);
    uiLayout     = new QVBoxLayout(this);
    systemLayout = new QVBoxLayout(this);
    customLayout = new QVBoxLayout(this);
    uiLayout->addLayout(systemLayout);
    uiLayout->addLayout(customLayout);
    uiLayout->addStretch();
    uiLayout->setSpacing(40);
    uiLayout->setMargin(0);
    initUi();
}

ShortcutUi::~ShortcutUi()
{

}

void ShortcutUi::initUi()
{
    initSystem();
    initCustom();
}

void ShortcutUi::initSystem()
{
    systemTitle         = new TitleLabel(this);
    systemShortcutGroup = new SettingGroup(this);
    //~ contents_path /Shortcut/System Shortcut
    systemTitle->setText(tr("System Shortcut"));
    systemLayout->addWidget(systemTitle);
    systemLayout->setSpacing(8);
    systemLayout->addWidget(systemShortcutGroup);
}

void ShortcutUi::initCustom()
{
    customTitle         = new TitleLabel(this);
    customShortcutGroup = new SettingGroup(this);
    addButton           = new AddButton(this);
    QHBoxLayout *layout = new QHBoxLayout(this);
    //~ contents_path /Shortcut/Customize Shortcut
    customTitle->setText(tr("Customize Shortcut"));
    customLayout->setSpacing(0);
    customLayout->addWidget(customTitle);
    customLayout->addSpacing(8);
    customLayout->addWidget(customShortcutGroup);
    customLayout->addSpacing(1);
    customLayout->addLayout(layout);
    layout->addWidget(addButton);

    addButton->setRadiusType(UkccFrame::BorderRadiusStyle::Around);
    customShortcutGroup->setOneRadiusStyle(UkccFrame::BorderRadiusStyle::Top);
    customShortcutGroup->setLastRadiusStyle(UkccFrame::BorderRadiusStyle::None);

    connect(addButton, &AddButton::clicked, this, &ShortcutUi::addButtonClicked);

}

void ShortcutUi::updateGeneralShort(DoubleClickShortCut *shortCutEdit, QString key, QString value, bool isConflict)
{

    if (isConflict) {
        for (int i = 0; i < mSystemEntryList.count(); i++) {

            if (keyToLib(mSystemEntryList[i].valueStr).contains(keyToLib(value), Qt::CaseInsensitive) && !value.isEmpty()) {


                for (int j = 0; j < mGeneralLineEdit.size(); j++) {
                    if (mGeneralLineEdit[j]->property("shortKey").toString().contains(keyToLib(value), Qt::CaseInsensitive) &&
                            shortCutEdit != mGeneralLineEdit[j]) {

                        mGeneralLineEdit[j]->blockSignals(true);
                        mGeneralLineEdit[j]->setText(tr("Null"));
                        mGeneralLineEdit[j]->blockSignals(false);

                        mGeneralName[j]->blockSignals(true);
                        mGeneralName[j]->setText(tr("Null"));
                        mGeneralName[j]->blockSignals(false);
                    }
                    mSystemEntryList[i].valueStr = "Null";
                    Q_EMIT updateSystemShortcut(mSystemEntryList[i].keyStr, "Null");
                }
            }
        }

        for (int i = 0; i < mCustomEntryList.count(); i++) {

            if (keyToLib(mCustomEntryList[i].bindingStr).contains(keyToLib(value), Qt::CaseInsensitive)) {
                for (int j = 0; j < mCustomLineEdit.size(); j++) {

                    qDebug() << "自定义快捷键冲突"  << mCustomLineEdit[j]->property("shortKey").toString() <<mCustomEntryList[i].bindingStr;
                    if (mCustomLineEdit[j]->property("shortKey").toString().contains(mCustomEntryList[i].bindingStr, Qt::CaseInsensitive)) {
                        mCustomLineEdit[j]->blockSignals(true);
                        mCustomLineEdit[j]->setText(tr("Null"));
                        mCustomLineEdit[j]->blockSignals(false);

                        mCustomName[j]->blockSignals(true);
                        mCustomName[j]->setText(tr("Null"));
                        mCustomName[j]->blockSignals(false);
                    }
                }

                mCustomEntryList[i].bindingStr = "";
                Q_EMIT updateCustomShortcut(mCustomEntryList[i].gsPath.toLatin1(), "binding", "Null");
            }
        }
    }

    for (int i = 0; i < mSystemEntryList.count(); i++) {
        if (key == mSystemEntryList[i].keyStr) {
            mSystemEntryList[i].valueStr = keyToLib(value);
            shortCutEdit->setProperty("shortKey", keyToLib(value));
            Q_EMIT updateSystemShortcut(key, keyToLib(value));
        }
    }
}

void ShortcutUi::addSystemShortcut(const QString &name, const QString &shortcut, const QString &mediaKey, KeyEntryList *systemEntryList, KeyEntryList *customEntryList)
{
    mSystemEntryList = *systemEntryList;

    int editWidth = 286;
    QString keyStr = mediaKey;
    QString value = shortcut;
    QStringList twiceValues;

    UkccFrame              *shortCutFrame         = new UkccFrame(this);

    FixLabel               *nameLabel             = new FixLabel(shortCutFrame);
    ClickFixLabel          *bingdingLabel         = new ClickFixLabel(shortCutFrame);
    ClickFixLabel          *bingdingLabelTwice    = nullptr;

    DoubleClickShortCut    *bingdingLineEditTwice = nullptr;
    DoubleClickShortCut    *bingdingLineEdit      = new DoubleClickShortCut(&mSystemEntryList, &mCustomEntryList, shortCutFrame);

    QHBoxLayout            *gHorLayout            = new QHBoxLayout(shortCutFrame);
    QHBoxLayout            *lineEditLayout        = new QHBoxLayout;
    QHBoxLayout            *lineEditLayoutTwice   = new QHBoxLayout;

#ifndef Nile
    FixLabel *bindingLabel = new FixLabel(shortCutFrame);
    bindingLabel->setText(getShowShortcutString(shortcut));
    bindingLabel->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
#endif

    nameLabel->setText(name);
    gHorLayout->setContentsMargins(16, 8, 16, 8);

    bool isTwice = false;

    if (shortcut.contains("or")) {
        editWidth = 130;
        isTwice = true;
        twiceValues = shortcut.split("or");
        value = twiceValues.at(1);

        bingdingLineEditTwice = new DoubleClickShortCut(systemEntryList, customEntryList);
        bingdingLineEditTwice->setAttribute(Qt::WA_InputMethodEnabled, false);
        bingdingLabelTwice = new ClickFixLabel(shortCutFrame);
        bingdingLabelTwice->setStyleSheet("QLabel{background-color:palette(button);border-radius: 4px}");

        bingdingLineEditTwice->setFixedWidth(150);

        bingdingLabelTwice->setFixedWidth(150);
        bingdingLabelTwice->setText(getShowShortcutString(value));

        lineEditLayoutTwice->addWidget(bingdingLineEditTwice, 1);
        lineEditLayoutTwice->addWidget(bingdingLabelTwice, 1);
        bingdingLineEditTwice->setVisible(false);

        bingdingLineEditTwice->blockSignals(true);
        bingdingLineEditTwice->setText(getShowShortcutString(value));
        bingdingLineEditTwice->setKeySequence(QKeySequence(value));
        bingdingLineEditTwice->setProperty("shortKey", value);
        bingdingLineEditTwice->blockSignals(false);

        mAllKeyI18nName.insert(keyStr + "2", QString(name));

        mGeneralLineEdit.append(bingdingLineEditTwice);
        mGeneralName.append(bingdingLabelTwice);

        connect(bingdingLabelTwice,&ClickFixLabel::doubleClicked,this,[=](){
            bingdingLabelTwice->hide();
            bingdingLineEditTwice->show();
            bingdingLineEditTwice->setFocus();
        });

        connect(bingdingLineEditTwice,&DoubleClickShortCut::focusOut,this,[=](){
            bingdingLabelTwice->show();
            bingdingLineEditTwice->hide();
        });

        connect(bingdingLineEditTwice, &DoubleClickShortCut::shortcutChanged, this, [=]() {
            QString text = bingdingLineEditTwice->text();
            QString shortCutValue = bingdingLineEditTwice->keySequence().toString();
            if (text.contains("Start") && shortCutValue.contains("Meta")) {
                shortCutValue.replace("Meta", "Start");
                text.replace("Start", "Win");
            }
            updateGeneralShort(bingdingLineEditTwice, keyStr + "2", shortCutValue, false);
            bingdingLineEditTwice->blockSignals(true);
            bingdingLineEditTwice->setText(getShowShortcutString(bingdingLineEditTwice->text()));
            bingdingLineEditTwice->blockSignals(false);
            bingdingLabelTwice->setText(bingdingLineEditTwice->text());
        });

        connect(bingdingLineEditTwice, &DoubleClickShortCut::shortcutConflict, this, [=](){

            QString tipValue = conflictTip(bingdingLineEditTwice->conflictKey, bingdingLineEditTwice->conflictValue);
            QString text = bingdingLineEditTwice->text();
            QMessageBox msg;
            msg.setIcon(QMessageBox::Warning);
            msg.addButton(tr("Cancel"), QMessageBox::NoRole);
            msg.addButton(tr("Use"), QMessageBox::ApplyRole);
            msg.setText(tr("Shortcut key conflict, use it?"));
            msg.setInformativeText(QString(tr("%1 occuied, using this combination will invalidate %2")).
                                   arg(bingdingLineEditTwice->text()).arg(tipValue));
            int ret = msg.exec();

            if (ret == 1) {
                QString shortCutValue = bingdingLineEditTwice->keySequence().toString();
                if (text.contains("Start") && shortCutValue.contains("Meta")) {
                    shortCutValue.replace("Meta", "Start");
                    text.replace("Start", "Win");
                }
                updateGeneralShort(bingdingLineEditTwice, keyStr + "2", shortCutValue, true);
                bingdingLineEditTwice->blockSignals(true);
                bingdingLineEditTwice->setText(getShowShortcutString(text));
                bingdingLineEditTwice->blockSignals(false);
                bingdingLabelTwice->setText(getShowShortcutString(text));
            } else {
                Q_EMIT bingdingLabelTwice->doubleClicked();
            }
        });

        connect(bingdingLineEditTwice, &DoubleClickShortCut::shortcutOccupy, this, [=]() {
            QMessageBox msg;
            msg.setIcon(QMessageBox::Warning);
            msg.setText(QString(tr("Shortcut \"%1\" occuied, please change the key combination")).arg(bingdingLineEditTwice->conflictKey));
            msg.exec();
            Q_EMIT bingdingLabelTwice->doubleClicked();
        });
    }

    if (isTwice) value = twiceValues.at(0);

    mGeneralLineEdit.append(bingdingLineEdit);
    mGeneralName.append(bingdingLabel);

    bingdingLabel->setText(getShowShortcutString(value));
    bingdingLabel->setFixedSize(editWidth, 36);
    bingdingLabel->setStyleSheet("QLabel{background-color:palette(button);border-radius: 4px}");
    bingdingLabel->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);

    bingdingLineEdit->setAttribute(Qt::WA_InputMethodEnabled, false);
    bingdingLineEdit->setFixedSize(editWidth, 36);
    bingdingLineEdit->setAlignment(Qt::AlignLeft);
    bingdingLineEdit->blockSignals(true);
    bingdingLineEdit->setText(getShowShortcutString(value));
    bingdingLineEdit->setKeySequence(QKeySequence(value));
    bingdingLineEdit->setProperty("shortKey", value);
    bingdingLineEdit->blockSignals(false);
    bingdingLineEdit->setVisible(false);

    QSizePolicy policy;
    policy = bingdingLabel->sizePolicy();
    policy.setHorizontalPolicy(QSizePolicy::Ignored);
    bingdingLabel->setSizePolicy(policy);

    policy = bingdingLineEdit->sizePolicy();
    policy.setHorizontalPolicy(QSizePolicy::Ignored);
    bingdingLineEdit->setSizePolicy(policy);

    policy = bingdingLabel->sizePolicy();
    policy.setHorizontalPolicy(QSizePolicy::Ignored);
    bingdingLabel->setSizePolicy(policy);

    mAllKeyI18nName.insert(keyStr, name);

    lineEditLayout->addWidget(bingdingLineEdit, 1);
    lineEditLayout->addWidget(bingdingLabel, 1);

    gHorLayout->addWidget(nameLabel, 1);

#ifdef Nile
    gHorLayout->addStretch();
    gHorLayout->addLayout(lineEditLayout, 1);
    gHorLayout->addLayout(lineEditLayoutTwice, 1);
#else
    bingdingLineEdit->setVisible(false);
    bingdingLabel->setVisible(false);
    if (bingdingLineEditTwice != nullptr) {
        bingdingLineEditTwice->setVisible(false);
        bingdingLabelTwice->setVisible(false);
    }
    gHorLayout->addWidget(bindingLabel, 1);
#endif

    shortCutFrame->setFixedHeight(52);
    shortCutFrame->setLayout(gHorLayout);

    systemShortcutGroup->addWidget(shortCutFrame);

    connect(bingdingLabel,&ClickFixLabel::doubleClicked,this,[=](){
        bingdingLabel->hide();
        bingdingLineEdit->show();
        bingdingLineEdit->setFocus();
    });

    connect(bingdingLineEdit,&DoubleClickShortCut::focusOut,this,[=](){
        bingdingLabel->show();
        bingdingLineEdit->hide();
    });

    connect(bingdingLineEdit, &DoubleClickShortCut::shortcutChanged, this, [=]() {
        QString text = bingdingLineEdit->text();
        QString shortCutValue = bingdingLineEdit->keySequence().toString();
        if (text.contains("Start") && shortCutValue.contains("Meta")) {
            shortCutValue.replace("Meta", "Start");
            text.replace("Start", "Win");
        }
        updateGeneralShort(bingdingLineEdit, keyStr, shortCutValue, false);
        bingdingLineEdit->blockSignals(true);
        bingdingLineEdit->setText(getShowShortcutString(text));
        bingdingLineEdit->blockSignals(false);
        bingdingLabel->setText(text);
    });

    connect(bingdingLineEdit, &DoubleClickShortCut::shortcutConflict, this, [=]() {
        QString tipValue = conflictTip(bingdingLineEdit->conflictKey, bingdingLineEdit->conflictValue);
        QString text = bingdingLineEdit->text();
        if (text.contains("Start")) {
            text.replace("Start", "Win");
        }
        if (text.contains("Meta")) {
            text.replace("Meta", "Win");
        }

        QMessageBox msg;
        msg.setIcon(QMessageBox::Warning);
        msg.addButton(tr("Cancel"), QMessageBox::NoRole);
        msg.addButton(tr("Use"), QMessageBox::ApplyRole);
        msg.setText(tr("Shortcut key conflict, use it?"));
        msg.setInformativeText(QString(tr("%1 occuied, using this combination will invalidate %2")).
                               arg(text).arg(tipValue));
        int ret = msg.exec();

        if (ret) {
            QString shortCutValue = bingdingLineEdit->keySequence().toString();
            if (text.contains("Start") && shortCutValue.contains("Meta")) {
                shortCutValue.replace("Meta", "Win");
                text.replace("Start", "Win");
            }
            updateGeneralShort(bingdingLineEdit, keyStr, shortCutValue, true);
            bingdingLineEdit->blockSignals(true);
            bingdingLineEdit->setText(getShowShortcutString(text));
            bingdingLineEdit->blockSignals(false);
            bingdingLabel->setText(getShowShortcutString(text));
        } else {
            Q_EMIT bingdingLabel->doubleClicked();
        }
    });

    connect(bingdingLineEdit, &DoubleClickShortCut::shortcutOccupy, this, [=]() {
        QMessageBox msg;
        msg.setIcon(QMessageBox::Warning);
        QString text = bingdingLineEdit->conflictKey;
        if (text.contains("Start")) {
            text.replace("Start", "Win");
        }
        if (text.contains("Meta")) {
            text.replace("Meta", "Win");
        }
        msg.setText(QString(tr("Shortcut \"%1\" occuied, please change the key combination")).arg(text));
        msg.exec();
        Q_EMIT bingdingLabel->doubleClicked();
    });
}

void ShortcutUi::addCustomShortcut(KeyEntry keyEntry, KeyEntryList *systemEntryList, KeyEntryList *customEntryList)
{
    mCustomEntryList = *customEntryList;

    UkccFrame           *frame            = new UkccFrame(this, UkccFrame::BorderRadiusStyle::Around, true);
    QHBoxLayout         *layout           = new QHBoxLayout(frame);
    QHBoxLayout         *lineEditLayout   = new QHBoxLayout;
    DoubleClickLineEdit *nameLineEdit     = new DoubleClickLineEdit(customEntryList, frame);
    DoubleClickShortCut *bingdingLineEdit = new DoubleClickShortCut(&mSystemEntryList, &mCustomEntryList);
    ClickFixLabel       *nameLabel        = new ClickFixLabel(frame);
    ClickFixLabel       *bingdingLabel    = new ClickFixLabel(frame);
    KeyEntry            *mKeyEntry        = new KeyEntry;

    mKeyEntry->gsSchema   = keyEntry.gsSchema;
    mKeyEntry->keyStr     = keyEntry.keyStr;
    mKeyEntry->valueStr   = keyEntry.valueStr;
    mKeyEntry->descStr    = keyEntry.descStr;
    mKeyEntry->gsPath     = keyEntry.gsPath;
    mKeyEntry->nameStr    = keyEntry.nameStr;
    mKeyEntry->bindingStr = keyEntry.bindingStr;
    mKeyEntry->actionStr  = keyEntry.actionStr;

    customShortcutGroup->addWidget(frame);
    addButton->setRadiusType(UkccFrame::BorderRadiusStyle::Bottom);

    // 大小写字母数字中文
    QRegExp rx("[a-zA-Z0-9\u4e00-\u9fa5]+");
    QRegExpValidator *regValidator = new QRegExpValidator(rx);
    // 输入限制
    nameLineEdit->setValidator(regValidator);
    nameLineEdit->setFixedHeight(36);
    bingdingLineEdit->setFixedSize(130,36);

    bingdingLineEdit->setAlignment(Qt::AlignRight);
    bingdingLabel->setAlignment(Qt::AlignRight | Qt::AlignCenter);
    bingdingLineEdit->setAttribute(Qt::WA_InputMethodEnabled, false);
    bingdingLabel->setFixedSize(130, 36);
    bingdingLabel->setStyleSheet("QLabel{background-color:palette(button);border-radius: 4px}");

    QToolButton *btn = new QToolButton(frame);
    btn->setAutoRaise(true);
    btn->setPopupMode(QToolButton::InstantPopup);
    QMenu *menu = new QMenu(btn);
    btn->setMenu(menu);
    QAction *edit = new QAction(btn);
    QAction *del = new QAction(btn);
    edit->setText(tr("Edit"));
    del->setText(tr("Delete"));
    menu->addAction(edit);
    menu->addAction(del);

    connect(del, &QAction::triggered, this, [=](){
        customShortcutGroup->removeWidget(frame);
        if (customShortcutGroup->showItemsCount() == 0) {
            addButton->setRadiusType(UkccFrame::BorderRadiusStyle::Around);
        }
        frame->deleteLater();
        Q_EMIT deleteShortcut(mKeyEntry->gsPath);
        for (int i = 0; i < customEntryList->size(); ++i) {
            if (customEntryList->at(i).nameStr == mKeyEntry->nameStr) {
                customEntryList->removeAt(i);
                break;
            }
        }
        delete mKeyEntry;
    });

    connect(edit, &QAction::triggered, this, [=](){
        AddShortcutDialog *addDialog = new AddShortcutDialog(systemEntryList, customEntryList, this);
        addDialog->setAttribute(Qt::WA_DeleteOnClose);
        addDialog->setSourceEnable(false);
        addDialog->setWindowTitle(QObject::tr("Edit Shortcut"));

        addDialog->setExecText(mKeyEntry->actionStr);
        addDialog->setNameText(mKeyEntry->nameStr);
        addDialog->setKeyText(mKeyEntry->bindingStr);
        addDialog->setKeyIsAvailable(3);
        connect(addDialog, &AddShortcutDialog::shortcutInfoSignal, this,
                [=](QString path, QString name, QString exec, QString key){
            Q_UNUSED(path)
            Q_EMIT toCreateShortcut(mKeyEntry->gsPath, name, exec, key, false);
            nameLineEdit->setToolTip(name);
            nameLineEdit->blockSignals(true);
            nameLineEdit->setText(name);
            nameLineEdit->blockSignals(false);
            bingdingLineEdit->blockSignals(true);
            QString mKey = getShowShortcutString(addDialog->keyToLib(key));
            if (mKey.contains("Meta")) {
                mKey.replace("Meta", "Win");
            }
            if (mKey.contains("Print")) {
                mKey.replace("Print", "PrtSc");
            }
            if (mKey.contains("Prtsc")) {
                mKey.replace("Prtsc", "PrtSc");
            }
            bingdingLineEdit->setText(mKey);
            bingdingLineEdit->blockSignals(false);
            nameLabel->setText(name);
            bingdingLabel->setText(bingdingLineEdit->text());
        });
        addDialog->exec();
    });

    btn->setIcon(QIcon::fromTheme("view-more-horizontal-symbolic"));
    btn->setProperty("useButtonPalette", true);
    btn->setFixedSize(36, 36);

    layout->setContentsMargins(8, 0, 16, 0);
    layout->setSpacing(24);
    layout->addLayout(lineEditLayout);
    layout->addWidget(btn);

    lineEditLayout->setMargin(0);
    lineEditLayout->setSpacing(140);

    QSizePolicy policy;
    policy = nameLineEdit->sizePolicy();
    policy.setHorizontalPolicy(QSizePolicy::Ignored);
    nameLineEdit->setSizePolicy(policy);

    policy = nameLabel->sizePolicy();
    policy.setHorizontalPolicy(QSizePolicy::Ignored);
    nameLabel->setSizePolicy(policy);

    policy = bingdingLineEdit->sizePolicy();
    policy.setHorizontalPolicy(QSizePolicy::Ignored);
    bingdingLineEdit->setSizePolicy(policy);

    policy = bingdingLabel->sizePolicy();
    policy.setHorizontalPolicy(QSizePolicy::Ignored);
    bingdingLabel->setSizePolicy(policy);

    lineEditLayout->addWidget(nameLineEdit, 1);
    lineEditLayout->addWidget(nameLabel, 1);
    lineEditLayout->addWidget(bingdingLineEdit, 1);
    lineEditLayout->addWidget(bingdingLabel, 1);
    nameLineEdit->setVisible(false);
    bingdingLineEdit->setVisible(false);

    nameLineEdit->blockSignals(true);
    nameLineEdit->setText(mKeyEntry->nameStr);
    nameLineEdit->blockSignals(false);
    bingdingLineEdit->blockSignals(true);
    bingdingLineEdit->setText(getShowShortcutString(mKeyEntry->bindingStr));
    bingdingLineEdit->blockSignals(false);

    nameLabel->setText(mKeyEntry->nameStr);
    bingdingLabel->setText(getShowShortcutString(mKeyEntry->bindingStr));


    connect(nameLabel,&ClickFixLabel::doubleClicked,this,[=](){
        nameLabel->hide();
        nameLineEdit->show();
        nameLineEdit->setFocus();
    });

    connect(bingdingLabel,&ClickFixLabel::doubleClicked,this,[=](){
        bingdingLabel->hide();
        bingdingLineEdit->show();
        bingdingLineEdit->setFocus();
    });

    connect(nameLineEdit,&DoubleClickLineEdit::focusOut,this,[=](){
        nameLabel->show();
        nameLineEdit->hide();
    });

    connect(bingdingLineEdit,&DoubleClickShortCut::focusOut,this,[=](){
        bingdingLabel->show();
        bingdingLineEdit->hide();
    });

    connect(nameLineEdit, &DoubleClickLineEdit::strChanged, this, [=](){
        Q_EMIT toCreateShortcut(mKeyEntry->gsPath, nameLineEdit->text(), mKeyEntry->actionStr, mKeyEntry->bindingStr, false, false); //只修改
        nameLabel->setText(nameLineEdit->text());
        mKeyEntry->nameStr = nameLineEdit->text();
    });

    connect(bingdingLineEdit, &DoubleClickShortCut::shortcutChanged, this, [=](){
        Q_EMIT toCreateShortcut(mKeyEntry->gsPath, mKeyEntry->nameStr, mKeyEntry->actionStr, bingdingLineEdit->keySequence().toString(), false, true); //只修改
        bingdingLineEdit->blockSignals(true);
        bingdingLineEdit->setText(getShowShortcutString(bingdingLineEdit->text()));
        bingdingLineEdit->blockSignals(false);
        bingdingLabel->setText(bingdingLineEdit->text());
        for (int i = 0; i < customEntryList->size(); i++) {
            if (customEntryList->at(i).nameStr == mKeyEntry->nameStr) {
                mKeyEntry->keyStr = customEntryList->at(i).keyStr;
                break;
            }
        }
    });

    return;
}

QString ShortcutUi::getShowShortcutString(QString src)
{
    if (src.contains("Meta")) {
        src.replace("Meta", "Win");
    }
    if (src.contains("Start")) {
        src.replace("Start", "Win");
    }
    src.replace("<","");
    src.replace(">","   ");
    src.replace(" or ",tr(" or "));
    QStringList temp_list = src.split(" ");
    QString str;
    for(qint32 i = 0; i < temp_list.count(); i++) {
        str += temp_list.at(i)
                .left(1).toUpper() +
                temp_list.at(i)
                .mid(1, temp_list.at(i).length() - 1);
        str += " ";
    }
    str.replace("Or","or");
    return  str;
}

QString ShortcutUi::keyToLib(QString key)
{
    if (key.contains("Meta")) {
        key.replace("Meta", "Win");
    }

    if (key.contains("Start")) {
        key.replace("Start", "Win");
    }

    if (key.contains("Print", Qt::CaseInsensitive)) {
        key.replace("Print", "PrtSc", Qt::CaseInsensitive);
    }

    if (key.contains("+")) {
        QStringList keys = key.split("+");
        if (keys.count() == 2) {
            QString lower = keys.at(1);
            QString keyToLib = "<" + keys.at(0) + ">" + lower.toLower();
            return keyToLib;
        } else if (keys.count() == 3) {
            QString lower = keys.at(2);
            QString keyToLib = "<" + keys.at(0) + ">" + "<" + keys.at(1) + ">" + lower.toLower();
            return keyToLib;
        } else if (keys.count() == 4) {
            QString lower = keys.at(3);
            QString keyToLib = "<" + keys.at(0) + ">" + "<" + keys.at(1) + ">" + "<" + keys.at(2) + ">" + lower.toLower();
            return keyToLib;
        }
    }
    return key;
}

QString ShortcutUi::conflictTip(const QString &key, const QString &value)
{
    if (!value.isEmpty()) {
        return value;
    } else {
        QMap<QString, QString>::Iterator it;
        for (it = mAllKeyI18nName.begin(); it != mAllKeyI18nName.end(); it++) {
            qDebug() << Q_FUNC_INFO << it.key() << it.value();
            if (it.key() == key) {
                return it.value();
            }
        }
    }
    return value;
}

void ShortcutUi::clearCustomShorcutFrame()
{
    customShortcutGroup->removeAndDeleteAllWidget();
}
