// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause

#include "vtkOpenGLTextActor.h"

#include "vtkCoordinate.h"
#include "vtkObjectFactory.h"
#include "vtkOpenGLGL2PSHelper.h"
#include "vtkRenderWindow.h"
#include "vtkRenderer.h"
#include "vtkViewport.h"

VTK_ABI_NAMESPACE_BEGIN
vtkStandardNewMacro(vtkOpenGLTextActor);

//------------------------------------------------------------------------------
void vtkOpenGLTextActor::PrintSelf(std::ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//------------------------------------------------------------------------------
int vtkOpenGLTextActor::RenderOverlay(vtkViewport* viewport)
{
  // Render to GL2PS if capturing:
  vtkOpenGLGL2PSHelper* gl2ps = vtkOpenGLGL2PSHelper::GetInstance();
  if (gl2ps)
  {
    switch (gl2ps->GetActiveState())
    {
      case vtkOpenGLGL2PSHelper::Capture:
        return this->RenderGL2PS(viewport, gl2ps);
      case vtkOpenGLGL2PSHelper::Background:
        return 0; // No rendering.
      case vtkOpenGLGL2PSHelper::Inactive:
        break; // continue rendering.
    }
  }

  return this->Superclass::RenderOverlay(viewport);
}

//------------------------------------------------------------------------------
vtkOpenGLTextActor::vtkOpenGLTextActor() = default;

//------------------------------------------------------------------------------
vtkOpenGLTextActor::~vtkOpenGLTextActor() = default;

//------------------------------------------------------------------------------
int vtkOpenGLTextActor::RenderGL2PS(vtkViewport* viewport, vtkOpenGLGL2PSHelper* gl2ps)
{
  std::string input = (this->Input && this->Input[0]) ? this->Input : "";
  if (input.empty())
  {
    return 0;
  }

  vtkRenderer* ren = vtkRenderer::SafeDownCast(viewport);
  if (!ren)
  {
    vtkWarningMacro("Viewport is not a renderer.");
    return 0;
  }

  // Figure out position:
  vtkCoordinate* coord = this->GetActualPositionCoordinate();
  double* textPos2 = coord->GetComputedDoubleDisplayValue(ren);
  double pos[3];
  pos[0] = textPos2[0];
  pos[1] = textPos2[1];
  pos[2] = -1.;

  vtkTextProperty* tprop = this->GetScaledTextProperty();
  gl2ps->DrawString(input, tprop, pos, pos[2] + 1e-6, ren);

  return 1;
}
VTK_ABI_NAMESPACE_END
